<?php
namespace BayonePlugin\Widgets;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}
use Elementor\Core\Kits\Documents\Tabs\Global_Colors;
use Elementor\Core\Kits\Documents\Tabs\Global_Typography;
use Elementor\Widget_Base;
use Elementor\Group_Control_Border;
use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;
use Elementor\Core\Schemes\Typography;
use Elementor\Group_Control_Text_Shadow;
use Elementor\Utils;
use Elementor\Group_Control_Background;


/**
 * Elementor heading widget.
 *
 * Elementor widget that displays an eye-catching headlines.
 *
 * @since 1.0.0
 */
class Bayone_Heading extends Widget_Base {

	/**
	 * Get widget name.
	 *
	 * Retrieve heading widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'bayone-heading';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve heading widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Bayone Heading', 'bayone_plg' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve heading widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-t-letter';
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the heading widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'bayone-elements' ];
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'heading', 'title', 'text' ];
	}

	public function get_script_depends() {
		return ['bayone-mirror-heading'];
	}

	/**
	 * Register heading widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function _register_controls() {
		$this->start_controls_section(
			'section_title',
			[
				'label' => esc_html__( 'Title', 'bayone_plg' ),
			]
		);

		$this->add_control(
			'title',
			[
				'label' => esc_html__( 'Title', 'bayone_plg' ),
				'type' => Controls_Manager::TEXTAREA,
				'dynamic' => [
					'active' => true,
				],
				'placeholder' => esc_html__( 'Enter your title', 'bayone_plg' ),
				'default' => esc_html__( 'Add Your Heading Text Here', 'bayone_plg' ),
				'description' => esc_html__( 'Note: use <span>...</span> to highlight text & <strong>...</strong> to bold text', 'bayone_plg' ),
			]
		);

		$this->add_control(
            'res_br',
            [
				'label'         => __( 'Responsive br', 'bayone_plg' ),
				'type'          => Controls_Manager::SWITCHER,
				'label_on'      => __( 'Show', 'bayone_plg' ),
				'label_off'     => __( 'Hide', 'bayone_plg' ),
				'return_value'  => 'yes',
				'default'  		=> 'yes',
			]
        );

		$this->add_control(
            'vertical-rl',
            [
				'label'         => __( 'Vertical rl', 'bayone_plg' ),
				'type'          => Controls_Manager::SWITCHER,
				'label_on'      => __( 'On', 'bayone_plg' ),
				'label_off'     => __( 'Off', 'bayone_plg' ),
				'return_value'  => 'yes',
				'default'  		=> 'no',
			]
		);

		$this->add_control(
        	'sticky',
            [
				'label'         => __( 'Sticky', 'bayone_plg' ),
				'type'          => Controls_Manager::SWITCHER,
				'label_on'      => __( 'Yes', 'bayone_plg' ),
				'label_off'     => __( 'No', 'bayone_plg' ),
				'return_value'  => 'yes',
				'default'  		=> '',
			]
        );
        
		$this->add_control(
        	'rotate_animation',
            [
				'label'         => __( 'Rotate Animation', 'bayone_plg' ),
				'type'          => Controls_Manager::SWITCHER,
				'label_on'      => __( 'Yes', 'bayone_plg' ),
				'label_off'     => __( 'No', 'bayone_plg' ),
				'return_value'  => 'yes',
				'default'  		=> 'no',
			]
        );

		$this->add_control(
			'link',
			[
				'label' => esc_html__( 'Link', 'elementor' ),
				'type' => Controls_Manager::URL,
				'dynamic' => [
					'active' => true,
				],
				'default' => [
					'url' => '',
				],
				'separator' => 'before',
			]
		);

		$this->add_control(
            'after_icon',
            [
				'label'         => __( 'Icon', 'bayone_plg' ),
				'type'          => Controls_Manager::SWITCHER,
				'label_on'      => __( 'Yes', 'bayone_plg' ),
				'label_off'     => __( 'No', 'bayone_plg' ),
				'return_value'  => 'yes',
				'default'  		=> 'no',
			]
        );
		
		$this->add_control(
			'selected_icon_after',
			[
				'label' => esc_html__('Icon', 'bayone_plg'),
				'type' => Controls_Manager::ICONS,
				'fa4compatibility' => 'icon',
				'skin' => 'inline',
				'label_block' => false,
				'condition' => [
					'after_icon' => 'yes'
				]
			]
		);

		$this->add_control(
			'img_after',
			[
				'label' => __('Image After', 'bayone_plg'),
				'type' => Controls_Manager::MEDIA,
				'condition' => [
					'after_icon' => 'yes'
				]
			]
		);

		$this->add_control(
			'icon_position',
			[
				'label' => esc_html__( 'Icon Position', 'bayone_plg' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'left' => [
						'title' => esc_html__( 'Left', 'bayone_plg' ),
						'icon' => 'eicon-order-start',
					],
					'both' => [
						'title' => esc_html__( 'Both', 'bayone_plg' ),
						'icon' => 'eicon-grow',
					],
					'right' => [
						'title' => esc_html__( 'Right', 'bayone_plg' ),
						'icon' => 'eicon-order-end',
					],
				],
				'default' => 'right',
			]
		);

		$this->add_responsive_control(
			'align',
			[
				'label' => esc_html__( 'Alignment', 'bayone_plg' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'left' => [
						'title' => esc_html__( 'Left', 'bayone_plg' ),
						'icon' => 'eicon-text-align-left',
					],
					'center' => [
						'title' => esc_html__( 'Center', 'bayone_plg' ),
						'icon' => 'eicon-text-align-center',
					],
					'right' => [
						'title' => esc_html__( 'Right', 'bayone_plg' ),
						'icon' => 'eicon-text-align-right',
					],
					'justify' => [
						'title' => esc_html__( 'Justified', 'bayone_plg' ),
						'icon' => 'eicon-text-align-justify',
					],
				],
				'default' => '',
				'selectors' => [
					'{{WRAPPER}}' => 'text-align: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'html_tag',
			[
				'label' => __( 'HTML Tag', 'bayone_plg' ),
				'type' => Controls_Manager::SELECT,
				'options' => [
					'h1' => __( 'H1', 'bayone_plg' ),
					'h2' => __( 'H2', 'bayone_plg' ),
					'h3' => __( 'H3', 'bayone_plg' ),
					'h4' => __( 'H4', 'bayone_plg' ),
					'h5' => __( 'H5', 'bayone_plg' ),
					'h6' => __( 'H6', 'bayone_plg' ),
					'div' => __( 'div', 'bayone_plg' ),
					'span' => __( 'span', 'bayone_plg' ),
					'p' => __( 'P', 'bayone_plg' ),
				],
				'default' => 'h2',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_title_style',
			[
				'label' => esc_html__( 'Title', 'bayone_plg' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

		$this->start_controls_tabs('heading_style');
		$this->start_controls_tab(
			'normal_title',
			[
				'label' => esc_html__( 'Normal', 'bayone_plg' ),
			]
		);
		$this->add_control(
			'title_color',
			[
				'label' => esc_html__( 'Text Color', 'bayone_plg' ),
				'type' => Controls_Manager::COLOR,

				'selectors' => [
					'{{WRAPPER}} .bayone-heading' => 'color: {{VALUE}};',
					'{{WRAPPER}} .bayone-heading a' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'typography',

				'selector' => '{{WRAPPER}} .bayone-heading',
			]
		);
        
        $this->add_group_control(
			\Elementor\Group_Control_Text_Stroke::get_type(),
			[
				'name' => 'text_stroke',
				'selector' => '{{WRAPPER}} .bayone-heading',
				'separator' => 'before',
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'hover_title',
			[
				'label' => esc_html__( 'Hover', 'bayone_plg' ),
			]
		);

		$this->add_control(
			'title_color_hover',
			[
				'label' => esc_html__( 'Text Color', 'bayone_plg' ),
				'type' => Controls_Manager::COLOR,

				'selectors' => [
					'{{WRAPPER}} .bayone-heading:hover' => 'color: {{VALUE}};',
					'{{WRAPPER}} .bayone-heading a:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'typography_hover',

				'selector' => '{{WRAPPER}} .bayone-heading:hover, {{WRAPPER}} .bayone-heading a:hover',
			]
		);
        
        $this->add_group_control(
			\Elementor\Group_Control_Text_Stroke::get_type(),
			[
				'name' => 'text_stroke_hover',
				'selector' => '{{WRAPPER}} .bayone-heading:hover',
			]
		);
		$this->end_controls_tab();

		$this->end_controls_tabs();		

        $this->add_control(
			'divider',
			[
				'type' => \Elementor\Controls_Manager::DIVIDER,
			]
		);

		$this->add_control(
			'opacity',
			[
				'type'    => Controls_Manager::NUMBER,
				'label'   => esc_html__( 'Opacity', 'bayone_plg' ),
				'selectors' => [
					'{{WRAPPER}} .bayone-heading' => 'opacity: {{VALUE}}'
				]
			]
		);

		$this->add_group_control(
			Group_Control_Text_Shadow::get_type(),
			[
				'name' => 'text_shadow',
				'selector' => '{{WRAPPER}} .bayone-heading',
			]
		);

		$this->add_control(
			'blend_mode',
			[
				'label' => esc_html__( 'Blend Mode', 'bayone_plg' ),
				'type' => Controls_Manager::SELECT,
				'options' => [
					'' => esc_html__( 'Normal', 'bayone_plg' ),
					'multiply' => 'Multiply',
					'screen' => 'Screen',
					'overlay' => 'Overlay',
					'darken' => 'Darken',
					'lighten' => 'Lighten',
					'color-dodge' => 'Color Dodge',
					'saturation' => 'Saturation',
					'color' => 'Color',
					'difference' => 'Difference',
					'exclusion' => 'Exclusion',
					'hue' => 'Hue',
					'luminosity' => 'Luminosity',
				],
				'selectors' => [
					'{{WRAPPER}} .bayone-heading' => 'mix-blend-mode: {{VALUE}}',
				],
				'separator' => 'none',
			]
		);
		$this->end_controls_section();

		//**********************************************************************************************************

        //*****Span Style*****//
		$this->start_controls_section(
			'span_styles',
			[
				'label' => __('Span Styles', 'bayone_plg'),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'span_typography',
				'label' => esc_html__('Span Text Typograpghy', 'bayone_plg'),
				'selector' => '{{WRAPPER}} span',
			]
		);

		$this->add_control(
			'span_text_color',
			[
				'label' => esc_html__( 'Span Text Color', 'bayone_plg' ),
				'type' => Controls_Manager::COLOR,
				'default' => '',
				'selectors' => [
					'{{WRAPPER}} span' => 'color: {{VALUE}};',
				],
			]
		);
        
        $this->add_group_control(
			\Elementor\Group_Control_Text_Stroke::get_type(),
			[
				'name' => '_span_text_stroke',
				'selector' => '{{WRAPPER}} .bayone-heading span',
			]
		);

		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name' => 'span_border',
				'selector' => '{{WRAPPER}} span',
				'separator' => 'before',
			]
		);

		$this->add_responsive_control(
			'span_padding',
			[
				'label' => esc_html__('Padding', 'bayone_plg'),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', '%', 'em', 'rem'],
				'selectors' => [
					'{{WRAPPER}} span' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'span_border_radius',
			[
				'label' => esc_html__('Border Radius', 'bayone_plg'),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', '%', 'em','rem'],
				'selectors' => [
					'{{WRAPPER}} span' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'span_opacity',
			[
				'type'    => Controls_Manager::NUMBER,
				'label'   => esc_html__( 'Opacity', 'bayone_plg' ),
				'selectors' => [
					'{{WRAPPER}} span' => 'opacity: {{VALUE}}'
				]
			]
		);
		
		$this->end_controls_section();

		$this->start_controls_section(
			'icon_style',
			[
				'label' => esc_html__('Icon Style', 'bayone_plg'),
				'tab' => Controls_Manager::TAB_STYLE
			]
		);
		$this->start_controls_tabs('after_icon_tabs');
		$this->start_controls_tab(
			'normal_icon',
			[
				'label' => __('Normal', 'bayone_plg')
			]
		);
		$this->add_control(
			'icon_color',
			[
				'label' => esc_html__('Icon Color', 'bayone_plg'),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .bayone-heading-title i' => 'color: {{VALUE}}; fill: {{VALUE}};',
					'{{WRAPPER}} .bayone-heading-title svg path' => ' fill: {{VALUE}};',
				]
			]
		);

        $left = is_rtl() ? 'right' : 'left';
        $right = is_rtl() ? 'left' : 'right';

		$this->add_control(
			'icon_indent',
			[
				'label' => esc_html__('Icon Spacing', 'bayone_plg'),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'max' => 50,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .bayone-heading-title .right ' => 'margin-'. $left .': {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .bayone-heading-title .left ' => 'margin-'. $right .': {{SIZE}}{{UNIT}};',
				],
			]
		);
		$this->add_responsive_control(
			'icon_size',
			[
				'label' => esc_html__('Icon Size', 'bayone_plg'),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'max' => 50,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .bayone-heading-title i' => 'font-size: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .bayone-heading-title svg ' => 'width: {{SIZE}}{{UNIT}} !important; height: {{SIZE}}{{UNIT}} !important',
					'{{WRAPPER}} .bayone-heading-title .icon-img' => 'width: {{SIZE}}{{UNIT}} !important; height: {{SIZE}}{{UNIT}} !important',
				],
			]
		);
		$this->add_responsive_control(
			'icon_line_height',
			[
				'label' => esc_html__('Icon Line Height', 'bayone_plg'),
				'type' => Controls_Manager::SLIDER,
				'selectors' => [
					'{{WRAPPER}} .bayone-heading-title .left' => 'height: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .bayone-heading-title .right ' => 'height: {{SIZE}}{{UNIT}} !important;',
				],
			]
		);

		$this->add_control(
			'icon_opacity',
			[
				'label' => __('Opacity', 'bayone_plg'),
				'type' => Controls_Manager::NUMBER,
				'default' => '1',
				'step' => '0.1',
				'max' => '1',
				'min' => '0',
				'selectors' => [
					'{{WRAPPER}} .bayone-heading-title i' => 'opacity: {{VALUE}};',
					'{{WRAPPER}} .bayone-heading-title svg ' => 'opacity: {{VALUE}}',
					'{{WRAPPER}} .bayone-heading-title .icon-img ' => 'opacity: {{VALUE}}',
				],
			]
		);
		$this->end_controls_tab();
		/////
		$this->start_controls_tab(
			'hover_after_icon',
			[
				'label' => __('Hover', 'bayone_plg')
			]
		);
		$this->add_control(
			'icon_color_hover',
			[
				'label' => esc_html__('Icon Color', 'bayone_plg'),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .bayone-heading-title a:hover i' => 'color: {{VALUE}} !important; fill: {{VALUE}} !important;',
					'{{WRAPPER}} .bayone-heading-title a:hover svg path' => ' fill: {{VALUE}} !important;',
				]
			]
		);
		$this->add_control(
			'icon_indent_hover',
			[
				'label' => esc_html__('Icon Spacing', 'bayone_plg'),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'max' => 50,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .bayone-heading-title a:hover .right ' => 'margin-left: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .bayone-heading-title a:hover .left ' => 'margin-right: {{SIZE}}{{UNIT}};',
				],
			]
		);
		$this->add_responsive_control(
			'icon_size_hover',
			[
				'label' => esc_html__('Icon Size', 'bayone_plg'),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'max' => 50,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .bayone-heading-title a:hover i' => 'font-size: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .bayone-heading-title a:hover svg ' => 'width: {{SIZE}}{{UNIT}} !important; height: {{SIZE}}{{UNIT}} !important',
					'{{WRAPPER}} .bayone-heading-title a:hover .icon-img' => 'opacity: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'icon_opacity_hover',
			[
				'label' => __('Opacity', 'bayone_plg'),
				'type' => Controls_Manager::NUMBER,
				'default' => '1',
				'step' => '0.1',
				'max' => '1',
				'min' => '0',
				'selectors' => [
					'{{WRAPPER}} .bayone-heading-title a:hover i' => 'opacity: {{VALUE}};',
					'{{WRAPPER}} .bayone-heading-title a:hover svg ' => 'opacity: {{VALUE}}',
					'{{WRAPPER}} .bayone-heading-title a:hover .icon-img ' => 'opacity: {{VALUE}}',
				],
			]
		);

		$this->end_controls_tab();
		$this->end_controls_tabs();

		$this->add_control(
			'icon_y',
			[
				'label' => esc_html__('Offset y', 'bayone_plg'),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'max' => 200,
						'min' => 0,
						'step' => 1,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .bayone-heading-title i' => 'bottom: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .bayone-heading-title svg ' => 'bottom: {{SIZE}}{{UNIT}}',
					'{{WRAPPER}} .bayone-heading-title .icon-img ' => 'bottom: {{SIZE}}{{UNIT}}',
				],
                'render_type' => 'ui',
			]
		);
		$this->end_controls_section();

	}

	/**
	 * Render heading widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

//		$bayone_gradient = 'yes' == $settings['text_additional_color'] ? ' bayone-additional-color' : '';
        $bayone_inline = 'yes' == $settings['after_icon'] || 'yes' == $settings['before_icon'] ? ' bayone-inline' : '';
		if ( '' === $settings['title'] ) {
			return;
		}
		$br = 'yes' == $settings['res_br'] ? ' hide-br' : '';
		$rotate_animation = 'yes' == $settings['rotate_animation'] ? ' d-rotate wow' : '';
		$rotate_animation_child = 'yes' == $settings['rotate_animation'] ? ' rotate-text' : '';
		$vertical_rl = 'yes' == $settings['vertical-rl'] ? ' vertical-rl' : '';

//		$this->add_render_attribute( 'title', 'class', 'bayone-heading'. $bayone_inline . $bayone_gradient . $br . $rotate_animation_child . $vertical_rl . '' );
        $this->add_render_attribute( 'title', 'class', 'bayone-heading'. $bayone_inline . $br . $rotate_animation_child . $vertical_rl . '' );

		if ( ! empty( $settings['size'] ) ) {
			$this->add_render_attribute( 'title', 'class', 'bayone-size-' . $settings['size'] );
		}

		$this->add_inline_editing_attributes( 'title' );

		$title = $settings['title'];

		$title_html = sprintf( '<%1$s %2$s>%3$s</%1$s>', Utils::validate_html_tag( $settings['html_tag'] ), $this->get_render_attribute_string( 'title' ), $title );
		// PHPCS - the variable $title_html holds safe data.
		// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		?>
		<div class="bayone-heading-title<?php echo $rotate_animation; ?>" <?php if($settings['sticky'] == 'yes') echo 'id="sticky_item"'; ?>>
			<?php if ( ! empty( $settings['link']['url'] ) ) : ?>
			<a href="<?php echo $settings['link']['url'] ?>">
			<?php endif;
			if($settings['icon_position'] == 'left' || $settings['icon_position'] == 'both'):
				if(!empty($settings['selected_icon_after']['value'])) :
				echo '<span class="left">';
					\Elementor\Icons_Manager::render_icon( $settings['selected_icon_after'], [ 'aria-hidden' => 'true' ] );
				echo '</span>';
				endif;
				if(!empty($settings['img_after']['url'])) : ?>
				<span class="icon-img left">
					<img src="<?php echo esc_url($settings['img_after']['url']); ?>" alt="heading-image">
				</span>
				<?php endif; 
			endif; ?>
			<?php echo $title_html;
			if($settings['icon_position'] == 'right' || $settings['icon_position'] == 'both'):
				if(!empty($settings['selected_icon_after']['value'])) :
					echo '<span class="right">';
						\Elementor\Icons_Manager::render_icon( $settings['selected_icon_after'], [ 'aria-hidden' => 'true' ] );
					echo '</span>';
				endif;
				if(!empty($settings['img_after']['url'])) : ?>
				<span class="icon-img right">
					<img src="<?php echo esc_url($settings['img_after']['url']); ?>" alt="heading-image">
				</span>
				<?php endif;
			endif; ?>
			<?php if ( ! empty( $settings['link']['url'] ) ) : ?>
			</a>
			<?php endif; ?>
		</div>
		<?php
	}

	/**
	 * Render heading widget output in the editor.
	 *
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 *
	 * @since 2.9.0
	 * @access protected
	 */
	protected function content_template() {

	}
}
