<?php

namespace ThemescampPlugin\Elementor;

if (!class_exists('Elementor\Group_Control_Background')) {
    return;
}

use Elementor\Controls_Manager;
use Elementor\Group_Control_Background;
use Elementor\Controls_Stack;
use Elementor\Plugin;
use Elementor\Group_Control_Base;

defined('ABSPATH') || exit(); // Exit if accessed directly

class TCG_Group_Background extends Group_Control_Background
{

    public $controls = [];
    protected static $fields;
    private static $background_types;

    public function array_insert_after(array $array, $key, array $new)
    {
        $keys = array_keys($array);
        $index = array_search($key, $keys);
        $pos = false === $index ? count($array) : $index + 1;
        return array_merge(array_slice($array, 0, $pos), $new, array_slice($array, $pos));
    }

    public function init_fields()
    {

        $active_breakpoints = Plugin::$instance->breakpoints->get_active_breakpoints();

        $location_device_args = [];
        $location_device_defaults = [
            'default' => [
                'unit' => '%',
            ],
        ];

        $angel_device_args = [];
        $angel_device_defaults = [
            'default' => [
                'unit' => 'deg',
            ],
        ];

        $position_device_args = [];
        $position_device_defaults = [
            'default' => 'center center',
        ];

        foreach ($active_breakpoints as $breakpoint_name => $breakpoint) {
            $location_device_args[$breakpoint_name] = $location_device_defaults;
            $angel_device_args[$breakpoint_name] = $angel_device_defaults;
            $position_device_args[$breakpoint_name] = $position_device_defaults;
        }

        $controls['color_c'] = [
            'label' => esc_html__('Third Color', 'elementor'),
            'type' => Controls_Manager::COLOR,
            'default' => '#f2295b',
            'render_type' => 'ui',
            'condition' => [
                'background' => ['tcg_gradient'],
            ],
            'of_type' => 'gradient',
        ];

        $controls['color_c_stop'] = [
            'label' => esc_html_x('Location', 'Background Control', 'elementor'),
            'type' => Controls_Manager::SLIDER,
            'size_units' => ['%', 'custom'],
            'default' => [
                'unit' => '%',
                'size' => 100,
            ],
            'device_args' => $location_device_args,
            'responsive' => true,
            'render_type' => 'ui',
            'condition' => [
                'background' => ['tcg_gradient'],
            ],
            'of_type' => 'gradient',
        ];

        $controls['tcg_gradient_angle'] = [
            'label' => esc_html__('Angle', 'elementor'),
            'type' => Controls_Manager::SLIDER,
            'size_units' => ['deg', 'grad', 'rad', 'turn', 'custom'],
            'default' => [
                'unit' => 'deg',
                'size' => 180,
            ],
            'device_args' => $angel_device_args,
            'responsive' => true,
            'selectors' => [
                '{{SELECTOR}}' => 'background-color: transparent; background-image: linear-gradient({{SIZE}}{{UNIT}}, {{color.VALUE}} {{color_stop.SIZE}}{{color_stop.UNIT}}, {{color_b.VALUE}} {{color_b_stop.SIZE}}{{color_b_stop.UNIT}}, {{color_c.VALUE}} {{color_c_stop.SIZE}}{{color_c_stop.UNIT}})',
            ],
            'condition' => [
                'background' => ['tcg_gradient'],
                'gradient_type' => 'linear',
            ],
            'of_type' => 'tcg_gradient',
        ];

        $controls['tcg_gradient_position'] = [
            'label' => esc_html__('Position', 'elementor'),
            'type' => Controls_Manager::SELECT,
            'options' => [
                'center center' => esc_html__('Center Center', 'elementor'),
                'center left' => esc_html__('Center Left', 'elementor'),
                'center right' => esc_html__('Center Right', 'elementor'),
                'top center' => esc_html__('Top Center', 'elementor'),
                'top left' => esc_html__('Top Left', 'elementor'),
                'top right' => esc_html__('Top Right', 'elementor'),
                'bottom center' => esc_html__('Bottom Center', 'elementor'),
                'bottom left' => esc_html__('Bottom Left', 'elementor'),
                'bottom right' => esc_html__('Bottom Right', 'elementor'),
            ],
            'default' => 'center center',
            'device_args' => $position_device_args,
            'responsive' => true,
            'selectors' => [
                '{{SELECTOR}}' => 'background-color: transparent; background-image: radial-gradient(at {{VALUE}}, {{color.VALUE}} {{color_stop.SIZE}}{{color_stop.UNIT}}, {{color_b.VALUE}} {{color_b_stop.SIZE}}{{color_b_stop.UNIT}}, {{color_c.VALUE}} {{color_c_stop.SIZE}}{{color_c_stop.UNIT}})',
            ],
            'condition' => [
                'background' => ['tcg_gradient'],
                'gradient_type' => 'radial',
            ],
            'of_type' => 'tcg_gradient',
        ];

        $fields = parent::init_fields(); // TODO: Change the autogenerated stub

        $fields = $this->array_insert_after(
            $fields,
            'color_b_stop',
            $controls
        );
        $modified_fields = [];

        foreach ($fields as $key => $field) {
            if (isset($field['condition']) && array_key_exists('background', $field['condition']) && in_array('gradient', $field['condition']['background']) && $key != 'gradient_position' && $key != 'gradient_angle') {
                array_push($field['condition']['background'], 'tcg_gradient');
                $modified_fields[$key] = $field;
            } else {
                $modified_fields[$key] = $field; // Add unmodified fields as well
            }
        }

        //var_dump($modified_fields);

        return $modified_fields;
    }

    protected function prepare_fields($fields)
    {
        $args = $this->get_args();

        $background_types = parent::get_background_types();

        // Add new background type 'gradient3'.
        $background_types['tcg_gradient'] = [
            'title' => esc_html__('3 Colors Gradient', 'elementor'),
            'icon' => 'eicon-barcode',
        ];

        $choose_types = [];

        foreach ($args['types'] as $type) {
            if (isset($background_types[$type])) {
                $choose_types[$type] = $background_types[$type];
            }
        }

        $fields['background']['options'] = $choose_types;

        //var_dump(Group_Control_Base::prepare_fields( $fields ));

        return Group_Control_Base::prepare_fields($fields);
    }
}


/**
 *  Elementor extra features
 */
class Themescamp_Extender
{

    public function __construct()
    {
        // extend Group_Control_Background
        add_action('elementor/controls/controls_registered', function () {

            $controls_manager = \Elementor\Plugin::$instance->controls_manager;
            $controls_manager->add_group_control('background', new TCG_Group_Background());
        });

        // add container 3 gradient colors background
        add_action('elementor/element/container/section_background/before_section_end', function ($stack) {

            $section_bg = Plugin::instance()->controls_manager->get_control_from_stack($stack->get_unique_name(), 'background_background');
            $section_bg['options']['tcg_gradient'] = [
                'title' => esc_html__('3 Colors Gradient', 'elementor'),
                'icon' => 'eicon-barcode',
            ];
            $stack->update_control('background_background', $section_bg);
        }, 10, 3);

        // add elements 3 gradient colors background
        add_action('elementor/element/common/_section_background/before_section_end', function ($stack) {

            $section_bg = Plugin::instance()->controls_manager->get_control_from_stack($stack->get_unique_name(), '_background_background');
            $section_bg['options']['tcg_gradient'] = [
                'title' => esc_html__('3 Colors Gradient', 'elementor'),
                'icon' => 'eicon-barcode',
            ];
            $stack->update_control('_background_background', $section_bg);
        }, 10, 3);

        // TCG select2
        add_action('elementor/controls/register', array($this, 'register_controls'));
        add_action('wp_ajax_tcg_select2_search_post', [$this, 'select2_ajax_posts_filter_autocomplete']);
        add_action('wp_ajax_tcg_select2_get_title', [$this, 'select2_ajax_get_posts_value_titles']);

        // elements controls
        add_action('elementor/element/before_section_end', [$this, 'register_tc_element_background_controls'], 10, 3);
        add_action('elementor/element/before_section_end', [$this, 'register_tc_element_border_controls'], 10, 3);
        add_action('elementor/element/after_section_end', [$this, 'register_tc_dark_mode_responsive_controls'], 10, 3);

        // container controls
        add_action('elementor/element/container/section_background/before_section_end', [$this, 'register_tc_container_background_controls'], 10, 3);
        add_action('elementor/element/container/section_background_overlay/before_section_end', [$this, 'register_tc_container_background_overlay_controls'], 10, 3);
        add_action('elementor/element/container/section_border/before_section_end', [$this, 'register_tc_container_border_controls'], 10, 3);

        // theme builder controls
        if ((isset($_GET['post']) && get_post_type($_GET['post']) === 'tcg_teb') || !isset($_GET['action'])) :
            add_action('elementor/element/container/section_background/after_section_end', [$this, 'register_tc_theme_builder_header_controls'], 10, 3);
        endif;

        // required assets for extending 
        add_action('elementor/frontend/section/before_render', [$this, 'should_script_enqueue']);
        add_action('elementor/frontend/container/before_render', [$this, 'should_script_enqueue']);
        add_action('elementor/frontend/widget/before_render', [$this, 'should_script_enqueue']);
        add_action('elementor/preview/enqueue_scripts', [$this, 'enqueue_scripts']);
    }

    public function get_name()
    {
        return 'tcg_extender';
    }

    public function tc_selectors_refactor(array $selectors, string $value)
    {
        foreach ($selectors as $key => $selector) {
            $selectors[$key] = $value;
        }
        return $selectors;
    }

    function register_tc_dark_mode_responsive_controls($widget, $widget_id, $args)
    {
        static $widgets = [
            '_section_responsive', /* Section */
        ];

        if (!in_array($widget_id, $widgets)) {
            return;
        }

        $widget->start_controls_section(
            'tc_dark_mode_responsive_section',
            [
                'label' => __('TCG Dark Mode Responsive', 'themescamp-core'),
                'tab' => Controls_Manager::TAB_ADVANCED,
            ]
        );

        $widget->add_control(
            'tc_dark_mode_responsive_hide_in_dark',
            [
                'label' => esc_html__('Hide in Dark Mode', 'themescamp-core'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Show', 'themescamp-core'),
                'label_off' => esc_html__('Hide', 'themescamp-core'),
                'return_value' => 'yes',
                'default' => 'no',
                'frontend_available' => true,
            ]
        );

        $widget->add_control(
            'tc_dark_mode_responsive_hide_in_light',
            [
                'label' => esc_html__('Hide in Light Mode', 'themescamp-core'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Show', 'themescamp-core'),
                'label_off' => esc_html__('Hide', 'themescamp-core'),
                'return_value' => 'yes',
                'default' => 'no',
                'frontend_available' => true,
            ]
        );

        $widget->end_controls_section();
    }

    function register_tc_theme_builder_header_controls($widget, $args)
    {

        $widget->start_controls_section(
            'tc_theme_builder_header_settings',
            [
                'label' => __('Header Settings', 'themescamp-core'),
                'tab' => Controls_Manager::TAB_LAYOUT,
            ]
        );

        $widget->add_control(
            'tc_theme_builder_header_sticky',
            [
                'label' => esc_html__('Sticky Header', 'themescamp-core'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '' => __('off', 'themescamp-core'),
                    'sticky' => __('Sticky', 'themescamp-core'),
                ],
                'separator' => 'before',
                'render_type' => 'none',
                'frontend_available' => true,
                'prefix_class' => 'tc-header-',
            ]
        );

        $widget->end_controls_section();
    }

    function register_tc_container_border_controls($widget, $args)
    {

        $widget->add_control(
            'tc_container_border_divider_dark_mode',
            [
                'type' => \Elementor\Controls_Manager::DIVIDER,
            ]
        );

        $widget->add_control(
            'tc_container_border_title_dark_mode',
            [
                'label' => esc_html__('TCG Dark Mode Border', 'themescamp-core'),
                'type' => \Elementor\Controls_Manager::HEADING,
                'separator' => 'none',
            ]
        );

        $widget->add_control(
            'tc_container_border_color_dark_mode',
            [
                'label' => esc_html__('Border Color (Dark Mode)', 'themescamp-core'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{WRAPPER}}' => 'border-color: {{VALUE}};',
                    '} body.tcg-dark-mode {{WRAPPER}}' => 'border-color: {{VALUE}};',
                ],
            ]
        );
    }

    function register_tc_container_background_controls($widget, $args)
    {

        $widget->add_control(
            'tc_container_background_blur_divider',
            [
                'type' => \Elementor\Controls_Manager::DIVIDER,
            ]
        );

        $widget->add_control(
            'tc_container_background_blur',
            [
                'label' => esc_html__('TCG Background Blur', 'themescamp-core'),
                'type' => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'max' => 250,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}}' => 'backdrop-filter: blur({{SIZE}}px);',
                ],
            ]
        );

        $widget->add_control(
            'tc_container_background_divider_dark_mode',
            [
                'type' => \Elementor\Controls_Manager::DIVIDER,
            ]
        );

        $widget->add_control(
            'tc_container_background_title_dark_mode',
            [
                'label' => esc_html__('TCG Dark Mode Background', 'themescamp-core'),
                'type' => \Elementor\Controls_Manager::HEADING,
                'separator' => 'none',
            ]
        );

        $widget->start_controls_tabs('_tabs_tc_container_background_dark_mode');

        $widget->start_controls_tab(
            '_tab_tc_container_background_normal_dark_mode',
            [
                'label' => esc_html__('Normal', 'themescamp-core'),
            ]
        );

        $widget->add_group_control(
            \Elementor\Group_Control_Background::get_type(),
            [
                'name' => 'tc_container_background_dark_mode',
                'selector' => '{{WRAPPER}}',
				'types' => [ 'classic', 'gradient', 'tcg_gradient' ],
                'fields_options' => [
                    'color' => [
                        'selectors' => [
                            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{SELECTOR}}' => 'background-color: {{VALUE}};',
                            '} body.tcg-dark-mode {{SELECTOR}}' => 'background-color: {{VALUE}};',
                        ],
                    ],
                    'gradient_angle' => [
                        'selectors' => [
                            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{SELECTOR}}' => 'background-color: transparent; background-image: linear-gradient({{SIZE}}{{UNIT}}, {{color.VALUE}} {{color_stop.SIZE}}{{color_stop.UNIT}}, {{color_b.VALUE}} {{color_b_stop.SIZE}}{{color_b_stop.UNIT}})',
                            '} body.tcg-dark-mode {{SELECTOR}}' => 'background-color: transparent; background-image: linear-gradient({{SIZE}}{{UNIT}}, {{color.VALUE}} {{color_stop.SIZE}}{{color_stop.UNIT}}, {{color_b.VALUE}} {{color_b_stop.SIZE}}{{color_b_stop.UNIT}})',
                        ],
                    ],
                    'gradient_position' => [
                        'selectors' => [
                            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{SELECTOR}}' => 'background-color: transparent; background-image: radial-gradient(at {{VALUE}}, {{color.VALUE}} {{color_stop.SIZE}}{{color_stop.UNIT}}, {{color_b.VALUE}} {{color_b_stop.SIZE}}{{color_b_stop.UNIT}})',
                            '} body.tcg-dark-mode {{SELECTOR}}' => 'background-color: transparent; background-image: radial-gradient(at {{VALUE}}, {{color.VALUE}} {{color_stop.SIZE}}{{color_stop.UNIT}}, {{color_b.VALUE}} {{color_b_stop.SIZE}}{{color_b_stop.UNIT}})',
                        ],
                    ],
                    'image' => [
                        'selectors' => [
                            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{SELECTOR}}' => 'background-image: url("{{URL}}");',
                            '} body.tcg-dark-mode {{SELECTOR}}' => 'background-image: url("{{URL}}");',
                        ],
                    ],
                    'position' => [
                        'selectors' => [
                            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{SELECTOR}}' => 'background-position: {{VALUE}};',
                            '} body.tcg-dark-mode {{SELECTOR}}' => 'background-position: {{VALUE}};',
                        ],
                    ],
                    'xpos' => [
                        'selectors' => [
                            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{SELECTOR}}' => 'background-position: {{SIZE}}{{UNIT}} {{ypos.SIZE}}{{ypos.UNIT}}',
                            '} body.tcg-dark-mode {{SELECTOR}}' => 'background-position: {{SIZE}}{{UNIT}} {{ypos.SIZE}}{{ypos.UNIT}}',
                        ],
                    ],
                    'ypos' => [
                        'selectors' => [
                            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{SELECTOR}}' => 'background-position: {{SIZE}}{{UNIT}} {{ypos.SIZE}}{{ypos.UNIT}}',
                            '} body.tcg-dark-mode {{SELECTOR}}' => 'background-position: {{SIZE}}{{UNIT}} {{ypos.SIZE}}{{ypos.UNIT}}',
                        ],
                    ],
                    'attachment' => [
                        'selectors' => [
                            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode (desktop+){{SELECTOR}}' => 'background-attachment: {{VALUE}};',
                            '} body.tcg-dark-mode (desktop+){{SELECTOR}}' => 'background-attachment: {{VALUE}};',
                        ],
                    ],
                    'repeat' => [
                        'selectors' => [
                            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{SELECTOR}}' => 'background-repeat: {{VALUE}};',
                            '} body.tcg-dark-mode {{SELECTOR}}' => 'background-repeat: {{VALUE}};',
                        ],
                    ],
                    'size' => [
                        'selectors' => [
                            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{SELECTOR}}' => 'background-size: {{VALUE}};',
                            '} body.tcg-dark-mode {{SELECTOR}}' => 'background-size: {{VALUE}};',
                        ],
                    ],
                    'bg_width' => [
                        'selectors' => [
                            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{SELECTOR}}' => 'background-size: {{SIZE}}{{UNIT}} auto',
                            '} body.tcg-dark-mode {{SELECTOR}}' => 'background-size: {{SIZE}}{{UNIT}} auto',
                        ],
                    ],
                    'video_fallback' => [
                        'selectors' => [
                            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{SELECTOR}}' => 'background: url("{{URL}}") 50% 50%; background-size: cover;',
                            '} body.tcg-dark-mode {{SELECTOR}}' => 'background: url("{{URL}}") 50% 50%; background-size: cover;',
                        ],
                    ],
                ]
            ]
        );

        $widget->end_controls_tab();

        $widget->start_controls_tab(
            '_tab_tc_container_background_hover_dark_mode',
            [
                'label' => esc_html__('Hover', 'themescamp-core'),
            ]
        );

        $widget->add_group_control(
            \Elementor\Group_Control_Background::get_type(),
            [
                'name' => 'tc_container_background_hover_dark_mode',
                'selector' => '{{WRAPPER}}:hover',
				'types' => [ 'classic', 'gradient', 'tcg_gradient' ],
                'fields_options' => [
                    'color' => [
                        'selectors' => [
                            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{SELECTOR}}' => 'background-color: {{VALUE}};',
                            '} body.tcg-dark-mode {{SELECTOR}}' => 'background-color: {{VALUE}};',
                        ],
                    ],
                    'gradient_angle' => [
                        'selectors' => [
                            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{SELECTOR}}' => 'background-color: transparent; background-image: linear-gradient({{SIZE}}{{UNIT}}, {{color.VALUE}} {{color_stop.SIZE}}{{color_stop.UNIT}}, {{color_b.VALUE}} {{color_b_stop.SIZE}}{{color_b_stop.UNIT}})',
                            '} body.tcg-dark-mode {{SELECTOR}}' => 'background-color: transparent; background-image: linear-gradient({{SIZE}}{{UNIT}}, {{color.VALUE}} {{color_stop.SIZE}}{{color_stop.UNIT}}, {{color_b.VALUE}} {{color_b_stop.SIZE}}{{color_b_stop.UNIT}})',
                        ],
                    ],
                    'gradient_position' => [
                        'selectors' => [
                            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{SELECTOR}}' => 'background-color: transparent; background-image: radial-gradient(at {{VALUE}}, {{color.VALUE}} {{color_stop.SIZE}}{{color_stop.UNIT}}, {{color_b.VALUE}} {{color_b_stop.SIZE}}{{color_b_stop.UNIT}})',
                            '} body.tcg-dark-mode {{SELECTOR}}' => 'background-color: transparent; background-image: radial-gradient(at {{VALUE}}, {{color.VALUE}} {{color_stop.SIZE}}{{color_stop.UNIT}}, {{color_b.VALUE}} {{color_b_stop.SIZE}}{{color_b_stop.UNIT}})',
                        ],
                    ],
                    'image' => [
                        'selectors' => [
                            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{SELECTOR}}' => 'background-image: url("{{URL}}");',
                            '} body.tcg-dark-mode {{SELECTOR}}' => 'background-image: url("{{URL}}");',
                        ],
                    ],
                    'position' => [
                        'selectors' => [
                            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{SELECTOR}}' => 'background-position: {{VALUE}};',
                            '} body.tcg-dark-mode {{SELECTOR}}' => 'background-position: {{VALUE}};',
                        ],
                    ],
                    'xpos' => [
                        'selectors' => [
                            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{SELECTOR}}' => 'background-position: {{SIZE}}{{UNIT}} {{ypos.SIZE}}{{ypos.UNIT}}',
                            '} body.tcg-dark-mode {{SELECTOR}}' => 'background-position: {{SIZE}}{{UNIT}} {{ypos.SIZE}}{{ypos.UNIT}}',
                        ],
                    ],
                    'ypos' => [
                        'selectors' => [
                            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{SELECTOR}}' => 'background-position: {{SIZE}}{{UNIT}} {{ypos.SIZE}}{{ypos.UNIT}}',
                            '} body.tcg-dark-mode {{SELECTOR}}' => 'background-position: {{SIZE}}{{UNIT}} {{ypos.SIZE}}{{ypos.UNIT}}',
                        ],
                    ],
                    'attachment' => [
                        'selectors' => [
                            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode (desktop+){{SELECTOR}}' => 'background-attachment: {{VALUE}};',
                            '} body.tcg-dark-mode (desktop+){{SELECTOR}}' => 'background-attachment: {{VALUE}};',
                        ],
                    ],
                    'repeat' => [
                        'selectors' => [
                            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{SELECTOR}}' => 'background-repeat: {{VALUE}};',
                            '} body.tcg-dark-mode {{SELECTOR}}' => 'background-repeat: {{VALUE}};',
                        ],
                    ],
                    'size' => [
                        'selectors' => [
                            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{SELECTOR}}' => 'background-size: {{VALUE}};',
                            '} body.tcg-dark-mode {{SELECTOR}}' => 'background-size: {{VALUE}};',
                        ],
                    ],
                    'bg_width' => [
                        'selectors' => [
                            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{SELECTOR}}' => 'background-size: {{SIZE}}{{UNIT}} auto',
                            '} body.tcg-dark-mode {{SELECTOR}}' => 'background-size: {{SIZE}}{{UNIT}} auto',
                        ],
                    ],
                    'video_fallback' => [
                        'selectors' => [
                            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{SELECTOR}}' => 'background: url("{{URL}}") 50% 50%; background-size: cover;',
                            '} body.tcg-dark-mode {{SELECTOR}}' => 'background: url("{{URL}}") 50% 50%; background-size: cover;',
                        ],
                    ],
                ]
            ]
        );

        $widget->end_controls_tab();

        $widget->end_controls_tabs();
    }

    function register_tc_container_background_overlay_controls($widget, $args)
    {

        $widget->add_control(
            'tc_container_background_overlay_divider_dark_mode',
            [
                'type' => \Elementor\Controls_Manager::DIVIDER,
            ]
        );

        $widget->add_control(
            'tc_container_background_overlay_title_dark_mode',
            [
                'label' => esc_html__('TCG Dark Mode Background Overlay', 'themescamp-core'),
                'type' => \Elementor\Controls_Manager::HEADING,
                'separator' => 'none',
            ]
        );

        $widget->start_controls_tabs('_tabs_tc_container_background_overlay');

        $widget->start_controls_tab(
            '_tab_tc_container_background_overlay_normal_dark_mode',
            [
                'label' => esc_html__('Normal', 'themescamp-core'),
            ]
        );

        $tc_container_background_overlay_selectors = [
            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{WRAPPER}}::before' => 'background-color: {{VALUE}};',
            '} body.tcg-dark-mode {{WRAPPER}}::before' => 'background-color: {{VALUE}};',

            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{WRAPPER}} > .elementor-background-video-container::before' => 'background-color: {{VALUE}};',
            '} body.tcg-dark-mode {{WRAPPER}} > .elementor-background-video-container::before' => 'background-color: {{VALUE}};',

            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{WRAPPER}} > .e-con-inner > .elementor-background-video-container::before' => 'background-color: {{VALUE}};',
            '} body.tcg-dark-mode {{WRAPPER}} > .e-con-inner > .elementor-background-video-container::before' => 'background-color: {{VALUE}};',

            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{WRAPPER}} > .e-con-inner > .elementor-background-slideshow::before' => 'background-color: {{VALUE}};',
            '} body.tcg-dark-mode {{WRAPPER}} > .e-con-inner > .elementor-background-slideshow::before' => 'background-color: {{VALUE}};',

            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{WRAPPER}} > .elementor-motion-effects-container > .elementor-motion-effects-layer::before' => 'background-color: {{VALUE}};',
            '} body.tcg-dark-mode {{WRAPPER}} > .elementor-motion-effects-container > .elementor-motion-effects-layer::before' => 'background-color: {{VALUE}};',
        ];

        $background_overlay_selector = '{{WRAPPER}}::before, {{WRAPPER}} > .elementor-background-video-container::before, {{WRAPPER}} > .e-con-inner > .elementor-background-video-container::before, {{WRAPPER}} > .elementor-background-slideshow::before, {{WRAPPER}} > .e-con-inner > .elementor-background-slideshow::before, {{WRAPPER}} > .elementor-motion-effects-container > .elementor-motion-effects-layer::before';

        $widget->add_group_control(
            \Elementor\Group_Control_Background::get_type(),
            [
                'name' => 'tc_container_background_overlay_dark_mode',
                'selector' => '{{WRAPPER}} > .elementor-element-populated >  .elementor-background-overlay',
				'types' => [ 'classic', 'gradient', 'tcg_gradient' ],
                'fields_options' => [
                    'background' => [
                        'selectors' => [
                            // Hack to set the `::before` content in order to render it only when there is a background overlay.
                            $background_overlay_selector => '--background-overlay: \'\';',
                        ],
                    ],
                    'color' => [
                        'selectors' => $this->tc_selectors_refactor($tc_container_background_overlay_selectors, 'background-color: {{VALUE}};'),
                    ],
                    'gradient_angle' => [
                        'selectors' => $this->tc_selectors_refactor($tc_container_background_overlay_selectors, 'background-color: transparent; background-image: linear-gradient({{SIZE}}{{UNIT}}, {{color.VALUE}} {{color_stop.SIZE}}{{color_stop.UNIT}}, {{color_b.VALUE}} {{color_b_stop.SIZE}}{{color_b_stop.UNIT}})'),
                    ],
                    'gradient_position' => [
                        'selectors' => $this->tc_selectors_refactor($tc_container_background_overlay_selectors, 'background-color: transparent; background-image: radial-gradient(at {{VALUE}}, {{color.VALUE}} {{color_stop.SIZE}}{{color_stop.UNIT}}, {{color_b.VALUE}} {{color_b_stop.SIZE}}{{color_b_stop.UNIT}})'),
                    ],
                    'image' => [
                        'selectors' => $this->tc_selectors_refactor($tc_container_background_overlay_selectors, 'background-image: url("{{URL}}");'),
                    ],
                    'position' => [
                        'selectors' => $this->tc_selectors_refactor($tc_container_background_overlay_selectors, 'background-position: {{VALUE}};'),
                    ],
                    'xpos' => [
                        'selectors' => $this->tc_selectors_refactor($tc_container_background_overlay_selectors, 'background-position: {{SIZE}}{{UNIT}} {{ypos.SIZE}}{{ypos.UNIT}}'),
                    ],
                    'ypos' => [
                        'selectors' => $this->tc_selectors_refactor($tc_container_background_overlay_selectors, 'background-position: {{xpos.SIZE}}{{xpos.UNIT}} {{SIZE}}{{UNIT}}'),
                    ],
                    'attachment' => [
                        'selectors' => $this->tc_selectors_refactor($tc_container_background_overlay_selectors, 'background-attachment: {{VALUE}};'),
                    ],
                    'repeat' => [
                        'selectors' => $this->tc_selectors_refactor($tc_container_background_overlay_selectors, 'background-repeat: {{VALUE}};'),
                    ],
                    'size' => [
                        'selectors' => $this->tc_selectors_refactor($tc_container_background_overlay_selectors, 'background-size: {{VALUE}};'),
                    ],
                    'bg_width' => [
                        'selectors' => $this->tc_selectors_refactor($tc_container_background_overlay_selectors, 'background-size: {{SIZE}}{{UNIT}} auto'),
                    ],
                    'video_fallback' => [
                        'selectors' => $this->tc_selectors_refactor($tc_container_background_overlay_selectors, 'background: url("{{URL}}") 50% 50%; background-size: cover;'),
                    ],
                ]
            ]
        );

        $widget->add_responsive_control(
            'tc_container_background_overlay_opacity_dark_mode',
            [
                'label' => esc_html__('Opacity', 'themescamp-core'),
                'type' => Controls_Manager::SLIDER,
                'default' => [
                    'size' => .5,
                ],
                'range' => [
                    'px' => [
                        'max' => 1,
                        'step' => 0.01,
                    ],
                ],
                'selectors' => $this->tc_selectors_refactor($tc_container_background_overlay_selectors, '--overlay-opacity: {{SIZE}};'),
                'condition' => [
                    'tc_container_background_overlay_dark_mode_background' => ['classic', 'gradient'],
                ],
            ]
        );

        $widget->end_controls_tab();

        $widget->start_controls_tab(
            '_tab_tc_container_background_overlay_hover_dark_mode',
            [
                'label' => esc_html__('Hover', 'themescamp-core'),
            ]
        );

        $tc_container_background_overlay_hover_selectors = [
            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{WRAPPER}}:hover::before' => 'background-color: {{VALUE}};',
            '} body.tcg-dark-mode {{WRAPPER}}:hover::before' => 'background-color: {{VALUE}};',

            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{WRAPPER}}:hover > .elementor-background-video-container::before' => 'background-color: {{VALUE}};',
            '} body.tcg-dark-mode {{WRAPPER}}:hover > .elementor-background-video-container::before' => 'background-color: {{VALUE}};',

            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{WRAPPER}}:hover > .e-con-inner > .elementor-background-video-container::before' => 'background-color: {{VALUE}};',
            '} body.tcg-dark-mode {{WRAPPER}}:hover > .e-con-inner > .elementor-background-video-container::before' => 'background-color: {{VALUE}};',

            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{WRAPPER}} > .e-con-inner > .elementor-background-slideshow:hover::before' => 'background-color: {{VALUE}};',
            '} body.tcg-dark-mode {{WRAPPER}} > .e-con-inner > .elementor-background-slideshow:hover::before' => 'background-color: {{VALUE}};',

            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{WRAPPER}} > .elementor-motion-effects-container > .elementor-motion-effects-layer:hover::before' => 'background-color: {{VALUE}};',
            '} body.tcg-dark-mode {{WRAPPER}} > .elementor-motion-effects-container > .elementor-motion-effects-layer:hover::before' => 'background-color: {{VALUE}};',
        ];

        $background_overlay_hover_selector = '{{WRAPPER}}:hover::before, {{WRAPPER}}:hover > .elementor-background-video-container::before, {{WRAPPER}}:hover > .e-con-inner > .elementor-background-video-container::before, {{WRAPPER}} > .elementor-background-slideshow:hover::before, {{WRAPPER}} > .e-con-inner > .elementor-background-slideshow:hover::before';

        $widget->add_group_control(
            \Elementor\Group_Control_Background::get_type(),
            [
                'name' => 'tc_container_background_overlay_hover_dark_mode',
                'selector' => '{{WRAPPER}}:hover > .elementor-element-populated >  .elementor-background-overlay',
				'types' => [ 'classic', 'gradient', 'tcg_gradient' ],
                'fields_options' => [
                    'background' => [
                        'selectors' => [
                            // Hack to set the `::before` content in order to render it only when there is a background overlay.
                            $background_overlay_hover_selector => '--background-overlay: \'\';',
                        ],
                    ],
                    'color' => [
                        'selectors' => $this->tc_selectors_refactor($tc_container_background_overlay_hover_selectors, 'background-color: {{VALUE}};'),
                    ],
                    'gradient_angle' => [
                        'selectors' => $this->tc_selectors_refactor($tc_container_background_overlay_hover_selectors, 'background-color: transparent; background-image: linear-gradient({{SIZE}}{{UNIT}}, {{color.VALUE}} {{color_stop.SIZE}}{{color_stop.UNIT}}, {{color_b.VALUE}} {{color_b_stop.SIZE}}{{color_b_stop.UNIT}})'),
                    ],
                    'gradient_position' => [
                        'selectors' => $this->tc_selectors_refactor($tc_container_background_overlay_hover_selectors, 'background-color: transparent; background-image: radial-gradient(at {{VALUE}}, {{color.VALUE}} {{color_stop.SIZE}}{{color_stop.UNIT}}, {{color_b.VALUE}} {{color_b_stop.SIZE}}{{color_b_stop.UNIT}})'),
                    ],
                    'image' => [
                        'selectors' => $this->tc_selectors_refactor($tc_container_background_overlay_hover_selectors, 'background-image: url("{{URL}}");'),
                    ],
                    'position' => [
                        'selectors' => $this->tc_selectors_refactor($tc_container_background_overlay_hover_selectors, 'background-position: {{VALUE}};'),
                    ],
                    'xpos' => [
                        'selectors' => $this->tc_selectors_refactor($tc_container_background_overlay_hover_selectors, 'background-position: {{SIZE}}{{UNIT}} {{ypos.SIZE}}{{ypos.UNIT}}'),
                    ],
                    'ypos' => [
                        'selectors' => $this->tc_selectors_refactor($tc_container_background_overlay_hover_selectors, 'background-position: {{xpos.SIZE}}{{xpos.UNIT}} {{SIZE}}{{UNIT}}'),
                    ],
                    'attachment' => [
                        'selectors' => $this->tc_selectors_refactor($tc_container_background_overlay_hover_selectors, 'background-attachment: {{VALUE}};'),
                    ],
                    'repeat' => [
                        'selectors' => $this->tc_selectors_refactor($tc_container_background_overlay_hover_selectors, 'background-repeat: {{VALUE}};'),
                    ],
                    'size' => [
                        'selectors' => $this->tc_selectors_refactor($tc_container_background_overlay_hover_selectors, 'background-size: {{VALUE}};'),
                    ],
                    'bg_width' => [
                        'selectors' => $this->tc_selectors_refactor($tc_container_background_overlay_hover_selectors, 'background-size: {{SIZE}}{{UNIT}} auto'),
                    ],
                    'video_fallback' => [
                        'selectors' => $this->tc_selectors_refactor($tc_container_background_overlay_hover_selectors, 'background: url("{{URL}}") 50% 50%; background-size: cover;'),
                    ],
                ]
            ]
        );

        $widget->add_responsive_control(
            'tc_container_background_overlay_opacity_hover_dark_mode',
            [
                'label' => esc_html__('Opacity', 'themescamp-core'),
                'type' => Controls_Manager::SLIDER,
                'default' => [
                    'size' => .5,
                ],
                'range' => [
                    'px' => [
                        'max' => 1,
                        'step' => 0.01,
                    ],
                ],
                'selectors' => $this->tc_selectors_refactor($tc_container_background_overlay_hover_selectors, '--overlay-opacity: {{SIZE}};'),
                'condition' => [
                    'tc_container_background_overlay_hover_dark_mode_background' => ['classic', 'gradient'],
                ],
            ]
        );

        $widget->end_controls_tab();

        $widget->end_controls_tabs();
    }

    function register_tc_element_border_controls($widget, $widget_id, $args)
    {
        static $widgets = [
            '_section_border', /* Section */
        ];

        if (!in_array($widget_id, $widgets)) {
            return;
        }

        $widget->add_control(
            'tc_element_border_divider_dark_mode',
            [
                'type' => \Elementor\Controls_Manager::DIVIDER,
            ]
        );

        $widget->add_control(
            'tc_element_border_title_dark_mode',
            [
                'label' => esc_html__('TCG Dark Mode Border', 'themescamp-core'),
                'type' => \Elementor\Controls_Manager::HEADING,
                'separator' => 'none',
            ]
        );

        $widget->add_control(
            'tc_element_border_color_dark_mode',
            [
                'label' => esc_html__('Border Color (Dark Mode)', 'themescamp-core'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{WRAPPER}} > .elementor-widget-container' => 'border-color: {{VALUE}};',
                    '} body.tcg-dark-mode {{WRAPPER}} > .elementor-widget-container' => 'border-color: {{VALUE}};',
                ],
            ]
        );
    }

    function register_tc_element_background_controls($widget, $widget_id, $args)
    {
        static $widgets = [
            '_section_background', /* Section */
        ];

        if (!in_array($widget_id, $widgets)) {
            return;
        }

        $widget->add_control(
            'tc_element_background_divider_dark_mode',
            [
                'type' => \Elementor\Controls_Manager::DIVIDER,
            ]
        );

        $widget->add_control(
            'tc_element_background_title_dark_mode',
            [
                'label' => esc_html__('TCG Dark Mode Background', 'themescamp-core'),
                'type' => \Elementor\Controls_Manager::HEADING,
                'separator' => 'none',
            ]
        );

        $widget->start_controls_tabs('_tabs_tc_element_background_dark_mode');

        $widget->start_controls_tab(
            '_tab_tc_element_background_normal_dark_mode',
            [
                'label' => esc_html__('Normal', 'themescamp-core'),
            ]
        );

        $widget->add_group_control(
            \Elementor\Group_Control_Background::get_type(),
            [
                'name' => 'tc_element_background_dark_mode',
                'selector' => '{{WRAPPER}} > .elementor-widget-container',
				'types' => [ 'classic', 'gradient', 'tcg_gradient' ],
                'fields_options' => [
                    'color' => [
                        'selectors' => [
                            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{SELECTOR}}' => 'background-color: {{VALUE}};',
                            '} body.tcg-dark-mode {{SELECTOR}}' => 'background-color: {{VALUE}};',
                        ],
                    ],
                    'gradient_angle' => [
                        'selectors' => [
                            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{SELECTOR}}' => 'background-color: transparent; background-image: linear-gradient({{SIZE}}{{UNIT}}, {{color.VALUE}} {{color_stop.SIZE}}{{color_stop.UNIT}}, {{color_b.VALUE}} {{color_b_stop.SIZE}}{{color_b_stop.UNIT}})',
                            '} body.tcg-dark-mode {{SELECTOR}}' => 'background-color: transparent; background-image: linear-gradient({{SIZE}}{{UNIT}}, {{color.VALUE}} {{color_stop.SIZE}}{{color_stop.UNIT}}, {{color_b.VALUE}} {{color_b_stop.SIZE}}{{color_b_stop.UNIT}})',
                        ],
                    ],
                    'gradient_position' => [
                        'selectors' => [
                            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{SELECTOR}}' => 'background-color: transparent; background-image: radial-gradient(at {{VALUE}}, {{color.VALUE}} {{color_stop.SIZE}}{{color_stop.UNIT}}, {{color_b.VALUE}} {{color_b_stop.SIZE}}{{color_b_stop.UNIT}})',
                            '} body.tcg-dark-mode {{SELECTOR}}' => 'background-color: transparent; background-image: radial-gradient(at {{VALUE}}, {{color.VALUE}} {{color_stop.SIZE}}{{color_stop.UNIT}}, {{color_b.VALUE}} {{color_b_stop.SIZE}}{{color_b_stop.UNIT}})',
                        ],
                    ],
                    'image' => [
                        'selectors' => [
                            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{SELECTOR}}' => 'background-image: url("{{URL}}");',
                            '} body.tcg-dark-mode {{SELECTOR}}' => 'background-image: url("{{URL}}");',
                        ],
                    ],
                    'position' => [
                        'selectors' => [
                            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{SELECTOR}}' => 'background-position: {{VALUE}};',
                            '} body.tcg-dark-mode {{SELECTOR}}' => 'background-position: {{VALUE}};',
                        ],
                    ],
                    'xpos' => [
                        'selectors' => [
                            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{SELECTOR}}' => 'background-position: {{SIZE}}{{UNIT}} {{ypos.SIZE}}{{ypos.UNIT}}',
                            '} body.tcg-dark-mode {{SELECTOR}}' => 'background-position: {{SIZE}}{{UNIT}} {{ypos.SIZE}}{{ypos.UNIT}}',
                        ],
                    ],
                    'ypos' => [
                        'selectors' => [
                            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{SELECTOR}}' => 'background-position: {{SIZE}}{{UNIT}} {{ypos.SIZE}}{{ypos.UNIT}}',
                            '} body.tcg-dark-mode {{SELECTOR}}' => 'background-position: {{SIZE}}{{UNIT}} {{ypos.SIZE}}{{ypos.UNIT}}',
                        ],
                    ],
                    'attachment' => [
                        'selectors' => [
                            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode (desktop+){{SELECTOR}}' => 'background-attachment: {{VALUE}};',
                            '} body.tcg-dark-mode (desktop+){{SELECTOR}}' => 'background-attachment: {{VALUE}};',
                        ],
                    ],
                    'repeat' => [
                        'selectors' => [
                            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{SELECTOR}}' => 'background-repeat: {{VALUE}};',
                            '} body.tcg-dark-mode {{SELECTOR}}' => 'background-repeat: {{VALUE}};',
                        ],
                    ],
                    'size' => [
                        'selectors' => [
                            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{SELECTOR}}' => 'background-size: {{VALUE}};',
                            '} body.tcg-dark-mode {{SELECTOR}}' => 'background-size: {{VALUE}};',
                        ],
                    ],
                    'bg_width' => [
                        'selectors' => [
                            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{SELECTOR}}' => 'background-size: {{SIZE}}{{UNIT}} auto',
                            '} body.tcg-dark-mode {{SELECTOR}}' => 'background-size: {{SIZE}}{{UNIT}} auto',
                        ],
                    ],
                    'video_fallback' => [
                        'selectors' => [
                            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{SELECTOR}}' => 'background: url("{{URL}}") 50% 50%; background-size: cover;',
                            '} body.tcg-dark-mode {{SELECTOR}}' => 'background: url("{{URL}}") 50% 50%; background-size: cover;',
                        ],
                    ],
                ]
            ]
        );

        $widget->end_controls_tab();

        $widget->start_controls_tab(
            '_tab_tc_element_background_hover_dark_mode',
            [
                'label' => esc_html__('Hover', 'themescamp-core'),
            ]
        );

        $widget->add_group_control(
            \Elementor\Group_Control_Background::get_type(),
            [
                'name' => 'tc_element_background_hover_dark_mode',
                'selector' => '{{WRAPPER}} > .elementor-widget-container:hover',
				'types' => [ 'classic', 'gradient', 'tcg_gradient' ],
                'fields_options' => [
                    'color' => [
                        'selectors' => [
                            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{SELECTOR}}' => 'background-color: {{VALUE}};',
                            '} body.tcg-dark-mode {{SELECTOR}}' => 'background-color: {{VALUE}};',
                        ],
                    ],
                    'gradient_angle' => [
                        'selectors' => [
                            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{SELECTOR}}' => 'background-color: transparent; background-image: linear-gradient({{SIZE}}{{UNIT}}, {{color.VALUE}} {{color_stop.SIZE}}{{color_stop.UNIT}}, {{color_b.VALUE}} {{color_b_stop.SIZE}}{{color_b_stop.UNIT}})',
                            '} body.tcg-dark-mode {{SELECTOR}}' => 'background-color: transparent; background-image: linear-gradient({{SIZE}}{{UNIT}}, {{color.VALUE}} {{color_stop.SIZE}}{{color_stop.UNIT}}, {{color_b.VALUE}} {{color_b_stop.SIZE}}{{color_b_stop.UNIT}})',
                        ],
                    ],
                    'gradient_position' => [
                        'selectors' => [
                            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{SELECTOR}}' => 'background-color: transparent; background-image: radial-gradient(at {{VALUE}}, {{color.VALUE}} {{color_stop.SIZE}}{{color_stop.UNIT}}, {{color_b.VALUE}} {{color_b_stop.SIZE}}{{color_b_stop.UNIT}})',
                            '} body.tcg-dark-mode {{SELECTOR}}' => 'background-color: transparent; background-image: radial-gradient(at {{VALUE}}, {{color.VALUE}} {{color_stop.SIZE}}{{color_stop.UNIT}}, {{color_b.VALUE}} {{color_b_stop.SIZE}}{{color_b_stop.UNIT}})',
                        ],
                    ],
                    'image' => [
                        'selectors' => [
                            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{SELECTOR}}' => 'background-image: url("{{URL}}");',
                            '} body.tcg-dark-mode {{SELECTOR}}' => 'background-image: url("{{URL}}");',
                        ],
                    ],
                    'position' => [
                        'selectors' => [
                            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{SELECTOR}}' => 'background-position: {{VALUE}};',
                            '} body.tcg-dark-mode {{SELECTOR}}' => 'background-position: {{VALUE}};',
                        ],
                    ],
                    'xpos' => [
                        'selectors' => [
                            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{SELECTOR}}' => 'background-position: {{SIZE}}{{UNIT}} {{ypos.SIZE}}{{ypos.UNIT}}',
                            '} body.tcg-dark-mode {{SELECTOR}}' => 'background-position: {{SIZE}}{{UNIT}} {{ypos.SIZE}}{{ypos.UNIT}}',
                        ],
                    ],
                    'ypos' => [
                        'selectors' => [
                            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{SELECTOR}}' => 'background-position: {{SIZE}}{{UNIT}} {{ypos.SIZE}}{{ypos.UNIT}}',
                            '} body.tcg-dark-mode {{SELECTOR}}' => 'background-position: {{SIZE}}{{UNIT}} {{ypos.SIZE}}{{ypos.UNIT}}',
                        ],
                    ],
                    'attachment' => [
                        'selectors' => [
                            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode (desktop+){{SELECTOR}}' => 'background-attachment: {{VALUE}};',
                            '} body.tcg-dark-mode (desktop+){{SELECTOR}}' => 'background-attachment: {{VALUE}};',
                        ],
                    ],
                    'repeat' => [
                        'selectors' => [
                            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{SELECTOR}}' => 'background-repeat: {{VALUE}};',
                            '} body.tcg-dark-mode {{SELECTOR}}' => 'background-repeat: {{VALUE}};',
                        ],
                    ],
                    'size' => [
                        'selectors' => [
                            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{SELECTOR}}' => 'background-size: {{VALUE}};',
                            '} body.tcg-dark-mode {{SELECTOR}}' => 'background-size: {{VALUE}};',
                        ],
                    ],
                    'bg_width' => [
                        'selectors' => [
                            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{SELECTOR}}' => 'background-size: {{SIZE}}{{UNIT}} auto',
                            '} body.tcg-dark-mode {{SELECTOR}}' => 'background-size: {{SIZE}}{{UNIT}} auto',
                        ],
                    ],
                    'video_fallback' => [
                        'selectors' => [
                            '@media (prefers-color-scheme: dark){ body.tcg-auto-mode {{SELECTOR}}' => 'background: url("{{URL}}") 50% 50%; background-size: cover;',
                            '} body.tcg-dark-mode {{SELECTOR}}' => 'background: url("{{URL}}") 50% 50%; background-size: cover;',
                        ],
                    ],
                ]
            ]
        );

        $widget->end_controls_tab();

        $widget->end_controls_tabs();

        $widget->add_control(
            'tc_element_background_blur_divider',
            [
                'type' => \Elementor\Controls_Manager::DIVIDER,
            ]
        );

        $widget->add_control(
            'tc_element_background_blur',
            [
                'label' => esc_html__('TCG Background Blur', 'themescamp-core'),
                'type' => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'max' => 250,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} > .elementor-widget-container' => 'backdrop-filter: blur({{SIZE}}px);',
                ],
            ]
        );
    }

    public function register_controls($controls_manager)
    {
        $controls_manager->register_control('tcg-select2', new \ThemescampPlugin\Elementor\Controls\Select2());
    }

    public static function get_query_post_list($post_type = 'any', $limit = -1, $search = '', $meta_query = [])
    {
        global $wpdb;
        $where = '';
        $data = [];

        if (-1 == $limit) {
            $limit = '';
        } elseif (0 == $limit) {
            $limit = "limit 0,1";
        } else {
            $limit = $wpdb->prepare(" limit 0,%d", esc_sql($limit));
        }

        if ('any' === $post_type) {
            $in_search_post_types = get_post_types(['exclude_from_search' => false]);
            if (empty($in_search_post_types)) {
                $where .= ' AND 1=0 ';
            } else {
                $where .= " AND {$wpdb->posts}.post_type IN ('" . join("', '", array_map('esc_sql', $in_search_post_types)) . "')";
            }
        } elseif (!empty($post_type)) {
            $where .= $wpdb->prepare(" AND {$wpdb->posts}.post_type = %s", esc_sql($post_type));
        }

        if (!empty($search)) {
            $where .= $wpdb->prepare(" AND {$wpdb->posts}.post_title LIKE %s", '%' . esc_sql($search) . '%');
        }

        // Process Meta Query
        if (!empty($meta_query)) {
            $where_meta = [];
            foreach ($meta_query as $mq) {
                $meta_key = esc_sql($mq['key']);
                $meta_value = esc_sql($mq['value']);
                $meta_compare = isset($mq['compare']) ? esc_sql($mq['compare']) : '=';
                $where_meta[] = "{$wpdb->postmeta}.meta_key = '$meta_key' AND {$wpdb->postmeta}.meta_value $meta_compare '$meta_value'";
            }
            if (!empty($where_meta)) {
                $where .= " AND ( " . join(' OR ', $where_meta) . " )";
            }
        }

        $query = "select post_title,ID from $wpdb->posts LEFT JOIN $wpdb->postmeta ON $wpdb->posts.ID = $wpdb->postmeta.post_id where post_status = 'publish' $where $limit";
        $results = $wpdb->get_results($query);
        if (!empty($results)) {
            foreach ($results as $row) {
                $data[$row->ID] = $row->post_title;
            }
        }
        return $data;
    }


    public function select2_ajax_posts_filter_autocomplete()
    {
        $post_type   = 'post';
        $source_name = 'post_type';

        if (!empty($_POST['post_type'])) {
            $post_type = sanitize_text_field($_POST['post_type']);
        }

        if (!empty($_POST['source_name'])) {
            $source_name = sanitize_text_field($_POST['source_name']);
        }

        $search  = !empty($_POST['term']) ? sanitize_text_field($_POST['term']) : '';
        $results = $post_list = [];
        $meta_query = !empty($_POST['meta_query']) ? $_POST['meta_query'] : '';
        switch ($source_name) {
            case 'taxonomy':
                $args = [
                    'hide_empty' => false,
                    'orderby'    => 'name',
                    'order'      => 'ASC',
                    'search'     => $search,
                    'number'     => '5',
                ];

                if ($post_type !== 'all') {
                    $args['taxonomy'] = $post_type;
                }

                $post_list = wp_list_pluck(get_terms($args), 'name', 'term_id');
                break;
            case 'user':
                if (!current_user_can('list_users')) {
                    $post_list = [];
                    break;
                }

                $users = [];

                foreach (get_users(['search' => "*{$search}*"]) as $user) {
                    $user_id           = $user->ID;
                    $user_name         = $user->display_name;
                    $users[$user_id] = $user_name;
                }

                $post_list = $users;
                break;
            default:
                $post_list = $this->get_query_post_list($post_type, 10, $search, $meta_query);
        }

        if (!empty($post_list)) {
            foreach ($post_list as $key => $item) {
                $results[] = ['text' => $item, 'id' => $key];
            }
        }

        wp_send_json(['results' => $results]);
    }

    /**
     * Select2 Ajax Get Posts Value Titles
     * get selected value to show elementor editor panel in select2 ajax search box
     *
     * @access public
     * @return void
     * @since 4.0.0
     */
    public function select2_ajax_get_posts_value_titles()
    {

        if (empty($_POST['id'])) {
            wp_send_json_error([]);
        }

        if (empty(array_filter($_POST['id']))) {
            wp_send_json_error([]);
        }
        $ids         = array_map('intval', $_POST['id']);
        $source_name = !empty($_POST['source_name']) ? sanitize_text_field($_POST['source_name']) : '';

        switch ($source_name) {
            case 'taxonomy':
                $args = [
                    'hide_empty' => false,
                    'orderby'    => 'name',
                    'order'      => 'ASC',
                    'include'    => implode(',', $ids),
                ];

                if ($_POST['post_type'] !== 'all') {
                    $args['taxonomy'] = sanitize_text_field($_POST['post_type']);
                }

                $response = wp_list_pluck(get_terms($args), 'name', 'term_id');
                break;
            case 'user':
                $users = [];

                foreach (get_users(['include' => $ids]) as $user) {
                    $user_id           = $user->ID;
                    $user_name         = $user->display_name;
                    $users[$user_id] = $user_name;
                }

                $response = $users;
                break;
            default:
                $post_info = get_posts([
                    'post_type' => sanitize_text_field($_POST['post_type']),
                    'include'   => implode(',', $ids)
                ]);
                $response  = wp_list_pluck($post_info, 'post_title', 'ID');
        }

        if (!empty($response)) {
            wp_send_json_success(['results' => $response]);
        } else {
            wp_send_json_error([]);
        }
    }

    public function enqueue_scripts($type)
    {
    }


    public function should_script_enqueue($widget)
    {
    }
}
new Themescamp_Extender;
