<?php
namespace ThemescampPlugin\Elementor\Elements\Widgets;

use Elementor\Group_Control_Typography;
use Elementor\Core\Kits\Documents\Tabs\Global_Typography;
use Elementor\Core\Schemes\Typography;
use Elementor\Group_Control_Background;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Css_Filter;
use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Utils;
use Elementor\repeater;
use Elementor\Frontend;
use Elementor\Icons_Manager;
use Elementor\Core\Schemes;
use Elementor\Group_Control_Image_Size;
use Elementor\Scheme_Base;
use Elementor\Group_Control_Text_Shadow;


if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly



/**
 * @since 1.1.0
 */
class TCG_Dynamic_Post_Meta extends Widget_Base {

    /**
     * Get widget name.
     *
     * Retrieve icon list widget name.
     *
     * @since 1.0.0
     * @access public
     *
     * @return string Widget name.
     */
    public function get_name() {
        return 'tcg-dynamic-post-meta';
    }

    /**
     * Get widget title.
     *
     * Retrieve icon list widget title.
     *
     * @since 1.0.0
     * @access public
     *
     * @return string Widget title.
     */
    public function get_title() {
        return esc_html__( 'TC Dynamic Post Meta', 'themescamp-plugin' );
    }

    /**
     * Get widget icon.
     *
     * Retrieve icon list widget icon.
     *
     * @since 1.0.0
     * @access public
     *
     * @return string Widget icon.
     */
    public function get_icon() {
        return 'eicon-document-file';
    }

    /**
     * Get widget keywords.
     *
     * Retrieve the list of keywords the widget belongs to.
     *
     * @since 2.1.0
     * @access public
     *
     * @return array Widget keywords.
     */
    public function get_keywords() {
        return [ 'posts', 'blogs', 'portfolio' ];
    }

    /**
     * Retrieve the list of categories the widget belongs to.
     *
     * Used to determine where to display the widget in the editor.
     *
     * Note that currently Elementor supports only one category.
     * When multiple categories passed, Elementor uses the first one.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return array Widget categories.
     */
    public function get_categories()
    {
        return ['themescamp-elements'];
    }

    /**
     * Register icon list widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since 3.1.0
     * @access protected
     */

    protected function register_controls(){

        $this->start_controls_section(
            'tcg_section_post__filters',
            [
                'label' => __('Query', 'themescamp-plugin'),
            ]
        );

        $this->add_control(
            'data_type',
            [
                'label' => __('Data Type', 'themescamp-plugin'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'title' => 'Title',
                    'date' => 'Date',
                    'time' => 'Time',
                    'excerpt' => 'Excerpt',
                    'author' => 'Author',
                    'avatar' => 'Avatar',
                    'categories' => 'Categories',
                    'tags' => 'Tags',
                    'comments' => 'Comments',
                ],
                'default' => 'title',

            ]
        );

		$this->add_control(
			'meta_separator',
			[
				'label' => esc_html__( 'Categories/tags Separator', 'themescamp-core' ),
				'type' => Controls_Manager::TEXT,
				'default' => ' / ',
				'condition' => [
					'data_type' => ['categories', 'tags'],
				],
			]
		);

		$this->add_control(
			'date_format',
			[
				'label' => esc_html__( 'Date Format', 'themescamp-core' ),
				'type' => Controls_Manager::SELECT,
				'default' => 'default',
				'options' => [
					'default' => 'Default',
					'0' => _x( 'March 6, 2018 (F j, Y)', 'Date Format', 'themescamp-core' ),
					'1' => '2018-03-06 (Y-m-d)',
					'2' => '03/06/2018 (m/d/Y)',
					'3' => '06/03/2018 (d/m/Y)',
					'custom' => esc_html__( 'Custom', 'themescamp-core' ),
				],
				'condition' => [
					'data_type' => 'date',
				],
			]
		);

		$this->add_control(
			'custom_date_format',
			[
				'label' => esc_html__( 'Custom Date Format', 'themescamp-core' ),
				'type' => Controls_Manager::TEXT,
				'default' => 'F j, Y',
				'condition' => [
					'data_type' => 'date',
					'date_format' => 'custom',
				],
				'description' => sprintf(
					/* translators: %s: Allowed data letters (see: http://php.net/manual/en/function.date.php). */
					__( 'Use the letters: %s', 'themescamp-core' ),
					'l D d j S F m M n Y y'
				),
			]
		);

		$this->add_control(
			'time_format',
			[
				'label' => esc_html__( 'Time Format', 'themescamp-core' ),
				'type' => Controls_Manager::SELECT,
				'default' => 'default',
				'options' => [
					'default' => 'Default',
					'0' => '3:31 pm (g:i a)',
					'1' => '3:31 PM (g:i A)',
					'2' => '15:31 (H:i)',
					'custom' => esc_html__( 'Custom', 'themescamp-core' ),
				],
				'condition' => [
					'data_type' => 'time',
				],
			]
		);

        $this->add_control(
			'custom_time_format',
			[
				'label' => esc_html__( 'Custom Time Format', 'themescamp-core' ),
				'type' => Controls_Manager::TEXT,
				'default' => 'g:i a',
				'placeholder' => 'g:i a',
				'condition' => [
					'data_type' => 'time',
					'time_format' => 'custom',
				],
				'description' => sprintf(
					/* translators: %s: Allowed time letters (see: http://php.net/manual/en/function.time.php). */
					__( 'Use the letters: %s', 'themescamp-core' ),
					'g G H i a A'
				),
			]
		);

		$this->add_control(
			'excerpt',
			[
				'label' => __('Blog Excerpt Length', 'themescamp-core'),
				'type' => Controls_Manager::NUMBER,
				'default' => '150',
				'min' => 10,
				'condition' => [
					'data_type' => 'excerpt',
				],
			]
		);

		$this->add_control(
			'excerpt_after',
			[
				'label' => __('After Excerpt text/symbol', 'themescamp-core'),
				'type' => Controls_Manager::TEXT,
				'condition' => [
					'data_type' => 'excerpt',
				],
				'default' => '...',
			]
		);

		$this->add_control(
			'comments_custom_strings',
			[
				'label' => esc_html__( 'Custom Format', 'themescamp-core' ),
				'type' => Controls_Manager::SWITCHER,
				'default' => false,
				'condition' => [
					'data_type' => 'comments',
				],
			]
		);

		$this->add_control(
			'string_no_comments',
			[
				'label' => esc_html__( 'No Comments', 'themescamp-core' ),
				'type' => Controls_Manager::TEXT,
				'placeholder' => esc_html__( 'No Comments', 'themescamp-core' ),
				'condition' => [
					'comments_custom_strings' => 'yes',
					'data_type' => 'comments',
				],
			]
		);

		$this->add_control(
			'string_one_comment',
			[
				'label' => esc_html__( 'One Comment', 'themescamp-core' ),
				'type' => Controls_Manager::TEXT,
				'placeholder' => esc_html__( 'One Comment', 'themescamp-core' ),
				'condition' => [
					'comments_custom_strings' => 'yes',
					'data_type' => 'comments',
				],
			]
		);

		$this->add_control(
			'string_comments',
			[
				'label' => esc_html__( 'Comments', 'themescamp-core' ),
				'type' => Controls_Manager::TEXT,
				'placeholder' => esc_html__( '%s Comments', 'themescamp-core' ),
				'condition' => [
					'comments_custom_strings' => 'yes',
					'data_type' => 'comments',
				],
			]
		);

        $this->add_control(
            'header_size',
            [
                'label' => esc_html__( 'HTML Tag', 'themescamp-core' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'h1' => 'H1',
                    'h2' => 'H2',
                    'h3' => 'H3',
                    'h4' => 'H4',
                    'h5' => 'H5',
                    'h6' => 'H6',
                    'div' => 'div',
                    'span' => 'span',
                    'p' => 'p',
                ],
                'default' => 'h2',
            ]
        );

        $this->add_responsive_control(
            'align',
            [
                'label' => esc_html__( 'Alignment', 'themescamp-core' ),
                'type' => Controls_Manager::CHOOSE,
                'options' => [
                    'left' => [
                        'title' => esc_html__( 'Left', 'themescamp-core' ),
                        'icon' => 'eicon-text-align-left',
                    ],
                    'center' => [
                        'title' => esc_html__( 'Center', 'themescamp-core' ),
                        'icon' => 'eicon-text-align-center',
                    ],
                    'right' => [
                        'title' => esc_html__( 'Right', 'themescamp-core' ),
                        'icon' => 'eicon-text-align-right',
                    ],
                    'justify' => [
                        'title' => esc_html__( 'Justified', 'themescamp-core' ),
                        'icon' => 'eicon-text-align-justify',
                    ],
                ],
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}}' => 'text-align: {{VALUE}};',
                ],
            ]
        );

		$this->add_control(
			'link',
			[
				'label' => esc_html__( 'Link', 'themescamp-core' ),
				'type' => Controls_Manager::SWITCHER,
                'return_value' => 'yes',
				'default' => 'yes',
			]
		);

        $this->end_controls_section();

        $this->start_controls_section(
            'section_title_style',
            [
                'label' => esc_html__( 'Text', 'themescamp-core' ),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );
        $this->start_controls_tabs( 'heading_color' );
        $this->start_controls_tab( 'text_normal',
            [
                'label' => esc_html__( 'Normal', 'themescamp-core' ),
            ]
        );
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'typography',
                'global' => [
                    'default' => Global_Typography::TYPOGRAPHY_PRIMARY,
                ],
                'selector' => '{{WRAPPER}} .tcg-dynamic-post-meta-text',
            ]
        );
        $this->add_control(
            'title_color',
            [
                'label' => esc_html__( 'Text Color', 'themescamp-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .tcg-dynamic-post-meta-text' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .tcg-dynamic-post-meta-text a' => 'color: {{VALUE}};',
                ],
            ]
        );
        $this->add_group_control(
            \Elementor\Group_Control_Text_Stroke::get_type(),
            [
                'name' => 'text_stroke',
                'selector' => '{{WRAPPER}} .tcg-dynamic-post-meta-text',
            ]
        );

        $this->add_control(
            'heading_opacity',
            [
                'label' => esc_html__( 'Opacity', 'themescamp-core' ),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'max' => 1,
                        'min' => 0.10,
                        'step' => 0.01,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .tcg-dynamic-post-meta-text' => 'opacity: {{SIZE}};',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab( 'text_hover',
            [
                'label' => esc_html__( 'Hover', 'themescamp-core' ),
            ]
        );
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'hover_typography',
                'global' => [
                    'default' => Global_Typography::TYPOGRAPHY_PRIMARY,
                ],
                'selector' => '{{WRAPPER}} .tcg-dynamic-post-meta-text:hover',
            ]
        );
        $this->add_control(
            'hover_title_color',
            [
                'label' => esc_html__( 'Hover Text Color', 'themescamp-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .tcg-dynamic-post-meta-text:hover' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .tcg-dynamic-post-meta-text a:hover' => 'color: {{VALUE}};',
                ],
            ]
        );
        $this->add_group_control(
            \Elementor\Group_Control_Text_Stroke::get_type(),
            [
                'name' => 'text_stroke_hover',
                'selector' => '{{WRAPPER}} .tcg-dynamic-post-meta-text:hover',
            ]
        );
        $this->add_control(
            'heading_opacity_hover',
            [
                'label' => esc_html__( 'Opacity', 'themescamp-core' ),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'max' => 1,
                        'min' => 0.10,
                        'step' => 0.01,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .tcg-dynamic-post-meta-text:hover' => 'opacity: {{SIZE}};',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->add_control(
            'separator_border2',
            [
                'type' => Controls_Manager::DIVIDER,
                'style' => 'thick',
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Text_Shadow::get_type(),
            [
                'name' => 'text_shadow',
                'selector' => '{{WRAPPER}} .tcg-dynamic-post-meta-text',
            ]
        );

        $this->add_control(
            'blend_mode',
            [
                'label' => esc_html__( 'Blend Mode', 'themescamp-core' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '' => esc_html__( 'Normal', 'themescamp-core' ),
                    'multiply' => esc_html__( 'Multiply', 'themescamp-core' ),
                    'screen' => esc_html__( 'Screen', 'themescamp-core' ),
                    'overlay' => esc_html__( 'Overlay', 'themescamp-core' ),
                    'darken' => esc_html__( 'Darken', 'themescamp-core' ),
                    'lighten' => esc_html__( 'Lighten', 'themescamp-core' ),
                    'color-dodge' => esc_html__( 'Color Dodge', 'themescamp-core' ),
                    'saturation' => esc_html__( 'Saturation', 'themescamp-core' ),
                    'color' => esc_html__( 'Color', 'themescamp-core' ),
                    'difference' => esc_html__( 'Difference', 'themescamp-core' ),
                    'exclusion' => esc_html__( 'Exclusion', 'themescamp-core' ),
                    'hue' => esc_html__( 'Hue', 'themescamp-core' ),
                    'luminosity' => esc_html__( 'Luminosity', 'themescamp-core' ),
                ],
                'selectors' => [
                    '{{WRAPPER}} .tcg-dynamic-post-meta-text' => 'mix-blend-mode: {{VALUE}}',
                ],
                'separator' => 'none',
            ]
        );
        $this->end_controls_section();

        $this->start_controls_section(
			'section_style_image',
			[
				'label' => esc_html__( 'Image', 'themescamp-core' ),
				'tab'   => Controls_Manager::TAB_STYLE,
                'condition' => [
                    'data_type' => 'avatar',
                ]
			]
		);

		$this->add_responsive_control(
			'width',
			[
				'label' => esc_html__( 'Width', 'themescamp-core' ),
				'type' => Controls_Manager::SLIDER,
				'default' => [
					'unit' => '%',
				],
				'tablet_default' => [
					'unit' => '%',
				],
				'mobile_default' => [
					'unit' => '%',
				],
				'size_units' => [ 'px', '%', 'em', 'rem', 'vw', 'custom' ],
				'range' => [
					'%' => [
						'min' => 1,
						'max' => 100,
					],
					'px' => [
						'min' => 1,
						'max' => 1000,
					],
					'vw' => [
						'min' => 1,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .tcg-dynamic-post-meta-img' => 'width: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'space',
			[
				'label' => esc_html__( 'Max Width', 'themescamp-core' ),
				'type' => Controls_Manager::SLIDER,
				'default' => [
					'unit' => '%',
				],
				'tablet_default' => [
					'unit' => '%',
				],
				'mobile_default' => [
					'unit' => '%',
				],
				'size_units' => [ 'px', '%', 'em', 'rem', 'vw', 'custom' ],
				'range' => [
					'%' => [
						'min' => 1,
						'max' => 100,
					],
					'px' => [
						'min' => 1,
						'max' => 1000,
					],
					'vw' => [
						'min' => 1,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .tcg-dynamic-post-meta-img' => 'max-width: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'height',
			[
				'label' => esc_html__( 'Height', 'themescamp-core' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%', 'em', 'rem', 'vh', 'custom' ],
				'range' => [
					'px' => [
						'min' => 1,
						'max' => 500,
					],
					'vh' => [
						'min' => 1,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .tcg-dynamic-post-meta-img' => 'height: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'object-fit',
			[
				'label' => esc_html__( 'Object Fit', 'themescamp-core' ),
				'type' => Controls_Manager::SELECT,
				'condition' => [
					'height[size]!' => '',
				],
				'options' => [
					'' => esc_html__( 'Default', 'themescamp-core' ),
					'fill' => esc_html__( 'Fill', 'themescamp-core' ),
					'cover' => esc_html__( 'Cover', 'themescamp-core' ),
					'contain' => esc_html__( 'Contain', 'themescamp-core' ),
				],
				'default' => '',
				'selectors' => [
					'{{WRAPPER}} .tcg-dynamic-post-meta-img' => 'object-fit: {{VALUE}};',
				],
			]
		);

		$this->add_responsive_control(
			'object-position',
			[
				'label' => esc_html__( 'Object Position', 'themescamp-core' ),
				'type' => Controls_Manager::SELECT,
				'options' => [
					'center center' => esc_html__( 'Center Center', 'themescamp-core' ),
					'center left' => esc_html__( 'Center Left', 'themescamp-core' ),
					'center right' => esc_html__( 'Center Right', 'themescamp-core' ),
					'top center' => esc_html__( 'Top Center', 'themescamp-core' ),
					'top left' => esc_html__( 'Top Left', 'themescamp-core' ),
					'top right' => esc_html__( 'Top Right', 'themescamp-core' ),
					'bottom center' => esc_html__( 'Bottom Center', 'themescamp-core' ),
					'bottom left' => esc_html__( 'Bottom Left', 'themescamp-core' ),
					'bottom right' => esc_html__( 'Bottom Right', 'themescamp-core' ),
				],
				'default' => 'center center',
				'selectors' => [
					'{{WRAPPER}} .tcg-dynamic-post-meta-img' => 'object-position: {{VALUE}};',
				],
				'condition' => [
					'object-fit' => 'cover',
				],
			]
		);

		$this->add_control(
			'separator_panel_style',
			[
				'type' => Controls_Manager::DIVIDER,
				'style' => 'thick',
			]
		);

		$this->start_controls_tabs( 'image_effects' );

		$this->start_controls_tab( 'image_normal',
			[
				'label' => esc_html__( 'Normal', 'themescamp-core' ),
			]
		);

		$this->add_control(
			'opacity',
			[
				'label' => esc_html__( 'Opacity', 'themescamp-core' ),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'max' => 1,
						'min' => 0.10,
						'step' => 0.01,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .tcg-dynamic-post-meta-img' => 'opacity: {{SIZE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Css_Filter::get_type(),
			[
				'name' => 'css_filters',
				'selector' => '{{WRAPPER}} .tcg-dynamic-post-meta-img',
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab( 'image_hover',
			[
				'label' => esc_html__( 'Hover', 'themescamp-core' ),
			]
		);

		$this->add_control(
			'opacity_hover',
			[
				'label' => esc_html__( 'Opacity', 'themescamp-core' ),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'max' => 1,
						'min' => 0.10,
						'step' => 0.01,
					],
				],
				'selectors' => [
					'{{WRAPPER}}:hover .tcg-dynamic-post-meta-img' => 'opacity: {{SIZE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Css_Filter::get_type(),
			[
				'name' => 'css_filters_hover',
				'selector' => '{{WRAPPER}}:hover .tcg-dynamic-post-meta-img',
			]
		);

		$this->add_control(
			'background_hover_transition',
			[
				'label' => esc_html__( 'Transition Duration', 'themescamp-core' ) . ' (s)',
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 3,
						'step' => 0.1,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .tcg-dynamic-post-meta-img' => 'transition-duration: {{SIZE}}s',
				],
			]
		);

		$this->add_control(
			'hover_animation',
			[
				'label' => esc_html__( 'Hover Animation', 'themescamp-core' ),
				'type' => Controls_Manager::HOVER_ANIMATION,
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name' => 'image_border',
				'selector' => '{{WRAPPER}} .tcg-dynamic-post-meta-img',
				'separator' => 'before',
			]
		);

		$this->add_responsive_control(
			'image_border_radius',
			[
				'label' => esc_html__( 'Border Radius', 'themescamp-core' ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em', 'rem', 'custom' ],
				'selectors' => [
					'{{WRAPPER}} .tcg-dynamic-post-meta-img' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			[
				'name' => 'image_box_shadow',
				'exclude' => [
					'box_shadow_position',
				],
				'selector' => '{{WRAPPER}} .tcg-dynamic-post-meta-img',
			]
		);

		$this->end_controls_section();

    }

    protected function render(){
        $settings = $this->get_settings();
        
        switch ($settings['data_type']){
            case 'title':

                $link_start = '';
                $link_end = '';

                if($settings['link'] == 'yes'): $link_start = '<a href="'.esc_url(get_the_permalink()).'">'; $link_end = '</a>'; endif; 

                ?>
                    <<?= Utils::validate_html_tag( $settings['header_size'] ) ?> class="tcg-dynamic-post-meta-text"><?= $link_start ?><?php the_title(); ?><?= $link_end ?></<?= Utils::validate_html_tag( $settings['header_size'] ) ?>>
                <?php
                break;
            case 'date':
                $custom_date_format = empty( $settings['custom_date_format'] ) ? 'F j, Y' : $settings['custom_date_format'];

				$format_options = [
					'default' => 'F j, Y',
					'0' => 'F j, Y',
					'1' => 'Y-m-d',
					'2' => 'm/d/Y',
					'3' => 'd/m/Y',
					'custom' => $custom_date_format,
				];

                $this->add_render_attribute( 'title', 'class', 'tcg-dynamic-post-meta-text' );

                $title = get_the_time( $format_options[ $settings['date_format'] ] );$link_start = '';
                $link_end = '';

                if($settings['link'] == 'yes'): $link_start = '<a href="'.esc_url(get_day_link(get_post_time('Y'), get_post_time('m'), get_post_time('j'))).'">'; $link_end = '</a>'; endif; 

                $title_html = sprintf( '<%1$s %2$s>'.$link_start.'%3$s'.$link_end.'</%1$s>', Utils::validate_html_tag( $settings['header_size'] ), $this->get_render_attribute_string( 'title' ), $title );

                // PHPCS - the variable $title_html holds safe data.
                echo $title_html;

                break;
            case 'time':
                $custom_time_format = empty( $settings['custom_time_format'] ) ? 'g:i a' : $settings['custom_time_format'];

                $format_options = [
                    'default' => 'g:i a',
                    '0' => 'g:i a',
                    '1' => 'g:i A',
                    '2' => 'H:i',
                    'custom' => $custom_time_format,
                ];
                
                $this->add_render_attribute( 'title', 'class', 'tcg-dynamic-post-meta-text' );

                $title = get_the_time( $format_options[ $settings['time_format'] ] );

                $link_start = '';
                $link_end = '';

                if($settings['link'] == 'yes'): $link_start = '<a href="'.esc_url(get_day_link(get_post_time('Y'), get_post_time('m'), get_post_time('j'))).'">'; $link_end = '</a>'; endif; 

                $title_html = sprintf( '<%1$s %2$s>'.$link_start.'%3$s'.$link_end.'</%1$s>', Utils::validate_html_tag( $settings['header_size'] ), $this->get_render_attribute_string( 'title' ), $title );

                // PHPCS - the variable $title_html holds safe data.
                echo $title_html;

                break;
            case 'excerpt':

                $excerpt = get_the_excerpt();
                $excerpt = substr($excerpt, 0, $settings['excerpt']);
                
                $this->add_render_attribute( 'title', 'class', 'tcg-dynamic-post-meta-text' );

                $title = $excerpt.$settings['excerpt_after'];

                $title_html = sprintf( '<%1$s %2$s>%3$s</%1$s>', Utils::validate_html_tag( $settings['header_size'] ), $this->get_render_attribute_string( 'title' ), $title );

                // PHPCS - the variable $title_html holds safe data.
                echo $title_html;

                break;
            case 'comments':
                if ( comments_open() ) {
                    $default_strings = [
                        'string_no_comments' => esc_html__( 'No Comments', 'themescamp-core' ),
                        'string_one_comment' => esc_html__( '%s Comment', 'themescamp-core' ),
                        'string_comments' => esc_html__( '%s Comments', 'themescamp-core' ),
                    ];

                    if ( 'yes' === $settings['comments_custom_strings'] ) {
                        if ( ! empty( $settings['string_no_comments'] ) ) {
                            $default_strings['string_no_comments'] = $settings['string_no_comments'];
                        }

                        if ( ! empty( $settings['string_one_comment'] ) ) {
                            $default_strings['string_one_comment'] = $settings['string_one_comment'];
                        }

                        if ( ! empty( $settings['string_comments'] ) ) {
                            $default_strings['string_comments'] = $settings['string_comments'];
                        }
                    }

                    $num_comments = (int) get_comments_number(); // get_comments_number returns only a numeric value

                    $title = 'Comments will appear here';

                    if ( 0 === $num_comments ) {
                        $title = $default_strings['string_no_comments'];
                    } else {
                        $title = sprintf( _n( $default_strings['string_one_comment'], $default_strings['string_comments'], $num_comments, 'themescamp-core' ), $num_comments );
                    }
                
                    $this->add_render_attribute( 'title', 'class', 'tcg-dynamic-post-meta-text' );

                    $link_start = '';
                    $link_end = '';

                    if($settings['link'] == 'yes'): $link_start = '<a href="'.esc_url(get_comments_link()).'">'; $link_end = '</a>'; endif; 

                    $title_html = sprintf( '<%1$s %2$s>'.$link_start.'%3$s'.$link_end.'</%1$s>', Utils::validate_html_tag( $settings['header_size'] ), $this->get_render_attribute_string( 'title' ), $title );

                    // PHPCS - the variable $title_html holds safe data.
                    echo $title_html;
                }
                break;
            case 'avatar':
                $link_start = '';
                $link_end = '';

                if($settings['link'] == 'yes'): $link_start = '<a href="'.esc_url(get_author_posts_url( get_the_author_meta( 'ID' ) )).'">'; $link_end = '</a>'; endif; 

                echo $link_start."<img class='tcg-dynamic-post-meta-img' src='".get_avatar_url( get_the_author_meta( 'ID' ), 96 )."'>".$link_end;
                break;
            case 'author':
                
                $this->add_render_attribute( 'title', 'class', 'tcg-dynamic-post-meta-text' );
                
                $title = get_the_author_meta( 'display_name' );

                $link_start = '';
                $link_end = '';

                if($settings['link'] == 'yes'): $link_start = '<a href="'.esc_url(get_author_posts_url( get_the_author_meta( 'ID' ) )).'">'; $link_end = '</a>'; endif; 

                $title_html = sprintf( '<%1$s %2$s>'.$link_start.'%3$s'.$link_end.'</%1$s>', Utils::validate_html_tag( $settings['header_size'] ), $this->get_render_attribute_string( 'title' ), $title );

                // PHPCS - the variable $title_html holds safe data.
                echo $title_html;
                break;
            case 'categories':
                global $post;
                $taxonomy_names = get_post_taxonomies();
                $category = get_the_terms( $post->ID, $taxonomy_names[0] );     
                
                ?>
                    <<?= Utils::validate_html_tag( $settings['header_size'] ) ?> class="tcg-dynamic-post-meta-text">
                        <?php 
                        $cat_counter = 0;
                        foreach ( $category as $cat){
                            if($cat_counter >= 1) echo $settings['meta_separator'];
                            echo '<a href="'.esc_url( get_term_link($cat) ).'">'.$cat->name.'</a>';
                            $cat_counter++;
                        } ?>
                    </<?= Utils::validate_html_tag( $settings['header_size'] ) ?>>
                <?php

                break;
            case 'tags':
                global $post;
                $taxonomy_names = get_post_taxonomies();
                $tags = get_the_terms( $post->ID, $taxonomy_names[1] );  

                ?>
                    <<?= Utils::validate_html_tag( $settings['header_size'] ) ?> class="tcg-dynamic-post-meta-text">
                        <?php 
                        $tag_counter = 0;
                        if($tags){
                            foreach ( $tags as $tag){
                                if($tag_counter >= 1) echo $settings['meta_separator'];
                                echo '<a href="'.esc_url( get_term_link($tag) ).'">'.$tag->name.'</a>';
                                $tag_counter++;
                            }
                        } ?>
                    </<?= Utils::validate_html_tag( $settings['header_size'] ) ?>>
                <?php

            break;

        }

        ?>
        
        <?php
    }
}