<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Main Zocker Core Class
 *
 * The main class that initiates and runs the plugin.
 *
 * @since 1.0.0
 */
final class Zocker_Extension {

	/**
	 * Plugin Version
	 *
	 * @since 1.0.0
	 *
	 * @var string The plugin version.
	 */
	const VERSION = '1.0.0';

	/**
	 * Minimum Elementor Version
	 *
	 * @since 1.0.0
	 *
	 * @var string Minimum Elementor version required to run the plugin.
	 */
	const MINIMUM_ELEMENTOR_VERSION = '2.0.0';

	/**
	 * Minimum PHP Version
	 *
	 * @since 1.0.0
	 *
	 * @var string Minimum PHP version required to run the plugin.
	 */
	const MINIMUM_PHP_VERSION = '7.0';

	/**
	 * Instance
	 *
	 * @since 1.0.0
	 *
	 * @access private
	 * @static
	 *
	 * @var Elementor_Test_Extension The single instance of the class.
	 */
	private static $_instance = null;

	/**
	 * Instance
	 *
	 * Ensures only one instance of the class is loaded or can be loaded.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 * @static
	 *
	 * @return Elementor_Test_Extension An instance of the class.
	 */
	public static function instance() {

		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self();
		}
		return self::$_instance;

	}

	/**
	 * Constructor
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function __construct() {
		add_action( 'plugins_loaded', [ $this, 'init' ] );

	}


	/**
	 * Initialize the plugin
	 *
	 * Load the plugin only after Elementor (and other plugins) are loaded.
	 * Checks for basic plugin requirements, if one check fail don't continue,
	 * if all check have passed load the files required to run the plugin.
	 *
	 * Fired by `plugins_loaded` action hook.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function init() {

		// Check if Elementor installed and activated
		if ( ! did_action( 'elementor/loaded' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_missing_main_plugin' ] );
			return;
		}

		// Check for required Elementor version
		if ( ! version_compare( ELEMENTOR_VERSION, self::MINIMUM_ELEMENTOR_VERSION, '>=' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_minimum_elementor_version' ] );
			return;
		}

		// Check for required PHP version
		if ( version_compare( PHP_VERSION, self::MINIMUM_PHP_VERSION, '<' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_minimum_php_version' ] );
			return;
		}

		// Add Plugin actions
		add_action( 'elementor/widgets/widgets_registered', [ $this, 'init_widgets' ] );

        // Register widget scripts
		add_action( 'elementor/frontend/after_enqueue_scripts', [ $this, 'widget_scripts' ]);

        // category register
		add_action( 'elementor/elements/categories_registered',[ $this, 'zocker_elementor_widget_categories' ] );

	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have Elementor installed or activated.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function admin_notice_missing_main_plugin() {

		if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

		$message = sprintf(
			/* translators: 1: Plugin name 2: Elementor */
			esc_html__( '"%1$s" requires "%2$s" to be installed and activated.', 'zocker' ),
			'<strong>' . esc_html__( 'Zocker Core', 'zocker' ) . '</strong>',
			'<strong>' . esc_html__( 'Elementor', 'zocker' ) . '</strong>'
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );

	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have a minimum required Elementor version.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function admin_notice_minimum_elementor_version() {

		if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

		$message = sprintf(
			/* translators: 1: Plugin name 2: Elementor 3: Required Elementor version */
			esc_html__( '"%1$s" requires "%2$s" version %3$s or greater.', 'zocker' ),
			'<strong>' . esc_html__( 'Zocker Core', 'zocker' ) . '</strong>',
			'<strong>' . esc_html__( 'Elementor', 'zocker' ) . '</strong>',
			 self::MINIMUM_ELEMENTOR_VERSION
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );

	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have a minimum required PHP version.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function admin_notice_minimum_php_version() {

		if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

		$message = sprintf(
			/* translators: 1: Plugin name 2: PHP 3: Required PHP version */
			esc_html__( '"%1$s" requires "%2$s" version %3$s or greater.', 'zocker' ),
			'<strong>' . esc_html__( 'Zocker Core', 'zocker' ) . '</strong>',
			'<strong>' . esc_html__( 'PHP', 'zocker' ) . '</strong>',
			 self::MINIMUM_PHP_VERSION
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );

	}

	/**
	 * Init Widgets
	 *
	 * Include widgets files and register them
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function init_widgets() {

		// Include Widget files
		require_once( ZOCKER_ADDONS . '/widgets/section-title.php' );
		require_once( ZOCKER_ADDONS . '/widgets/about-us.php' );
		require_once( ZOCKER_ADDONS . '/widgets/blog-post.php' );
		require_once( ZOCKER_ADDONS . '/widgets/button.php' );
		require_once( ZOCKER_ADDONS . '/widgets/testimonial-slider.php' );
		require_once( ZOCKER_ADDONS . '/widgets/team-member.php' );
	 	require_once( ZOCKER_ADDONS . '/widgets/image.php' );
	 	require_once( ZOCKER_ADDONS . '/widgets/zocker-faq.php' );
	 	require_once( ZOCKER_ADDONS . '/widgets/video-slider.php' );
	 	require_once( ZOCKER_ADDONS . '/widgets/zocker-match-tab.php' );
	 	require_once( ZOCKER_ADDONS . '/widgets/players.php' );
	 	require_once( ZOCKER_ADDONS . '/widgets/client-logo.php' );
	 	require_once( ZOCKER_ADDONS . '/widgets/stream-time-table.php' );
	 	require_once( ZOCKER_ADDONS . '/widgets/video-streaming-tab.php' );
	 	require_once( ZOCKER_ADDONS . '/widgets/trending-news.php' );
	 	require_once( ZOCKER_ADDONS . '/widgets/trending-news-list-style.php' );
	 	require_once( ZOCKER_ADDONS . '/widgets/most-viewed-blog.php' );
	 	require_once( ZOCKER_ADDONS . '/widgets/recent-post-thumb.php' );
	 	require_once( ZOCKER_ADDONS . '/widgets/social-widget.php' );
	 	require_once( ZOCKER_ADDONS . '/widgets/top-games.php' );
	 	require_once( ZOCKER_ADDONS . '/widgets/zocker-newsletter.php' );
	 	require_once( ZOCKER_ADDONS . '/widgets/zocker-video-news.php' );
	 	require_once( ZOCKER_ADDONS . '/widgets/popular-news.php' );
	 	require_once( ZOCKER_ADDONS . '/widgets/zocker-categories.php' );
	 	require_once( ZOCKER_ADDONS . '/widgets/player-info.php' );
	 	require_once( ZOCKER_ADDONS . '/widgets/live-stream-box.php' );
	 	require_once( ZOCKER_ADDONS . '/widgets/zocker-team.php' );
	 	require_once( ZOCKER_ADDONS . '/widgets/zocker-tournament.php' );
	 	require_once( ZOCKER_ADDONS . '/widgets/google-map.php' );
	 	require_once( ZOCKER_ADDONS . '/widgets/product-slider.php' );
	 	require_once( ZOCKER_ADDONS . '/widgets/player-rank-tab.php' );

		// Header Elements
		require_once( ZOCKER_ADDONS . '/header/header.php' );

		// Footer Elements
		// require_once( ZOCKER_ADDONS . '/footer-widgets/newsletter-widget.php' );
		require_once( ZOCKER_ADDONS . '/footer-widgets/footer-news-feed.php' );

		// Register widget
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Zocker_Section_Title_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Zocker_About_Us() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Zocker_Blog_Post() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Zocker_Button() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Zocker_Testimonial_Slider() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Zocker_Team_Member() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Zocker_Image() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Zocker_Faq() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Zocker_Video_Slider() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Zocker_Match_Tab() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Zocker_Player() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Zocker_Client_Logo() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Zocker_Stream_Time_Table() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Zocker_Video_Streaming_Tab() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Zocker_Trending_News() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Zocker_Trending_News_List() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Zocker_Most_Viewed_Blog() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Zocker_Recent_News() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Zocker_Social_Icon() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Zocker_Top_Games() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Zocker_Newsletter() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Zocker_Video_News() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Zocker_Popular_News() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Zocker_Categories() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Zocker_Live_Stream_Box() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Zocker_Team() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Zocker_Tournaments() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Zocker_Google_Map() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Zocker_Product_Slider() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Zocker_Player_Rank_Tab() );

		// Header Widget Register
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Zocker_Header() );

		// Footer Widget Register
		// \Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Zocker_Newsletter_Widgets() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Zocker_Footer_News_Feed() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Zocker_Player_Info() );

	}

    public function widget_scripts() {
        wp_enqueue_script(
            'zocker-frontend-script',
            ZOCKER_PLUGDIRURI . 'assets/js/zocker-frontend.js',
            array('jquery'),
            false,
            true
		);
	}

    function zocker_elementor_widget_categories( $elements_manager ) {
        $elements_manager->add_category(
            'zocker',
            [
                'title' => __( 'Playrex Helper', 'zocker' ),
                'icon' 	=> 'fa fa-plug',
            ]
        );
        $elements_manager->add_category(
            'zocker_footer_elements',
            [
                'title' => __( 'Playrex Footer Elements', 'zocker' ),
                'icon' 	=> 'fa fa-plug',
            ]
		);

		$elements_manager->add_category(
            'zocker_header_elements',
            [
                'title' => __( 'Playrex Header Elements', 'zocker' ),
                'icon' 	=> 'fa fa-plug',
            ]
        );

	}

}

Zocker_Extension::instance();