<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Main Playrex Core Class
 *
 * The main class that initiates and runs the plugin.
 *
 * @since 1.0.0
 */
final class Playrex_Extension {

	/**
	 * Plugin Version
	 *
	 * @since 1.0.0
	 *
	 * @var string The plugin version.
	 */
	const VERSION = '1.0.0';

	/**
	 * Minimum Elementor Version
	 *
	 * @since 1.0.0
	 *
	 * @var string Minimum Elementor version required to run the plugin.
	 */
	const MINIMUM_ELEMENTOR_VERSION = '2.0.0';

	/**
	 * Minimum PHP Version
	 *
	 * @since 1.0.0
	 *
	 * @var string Minimum PHP version required to run the plugin.
	 */
	const MINIMUM_PHP_VERSION = '7.0';

	/**
	 * Instance
	 *
	 * @since 1.0.0
	 *
	 * @access private
	 * @static
	 *
	 * @var Elementor_Test_Extension The single instance of the class.
	 */
	private static $_instance = null;

	/**
	 * Instance
	 *
	 * Ensures only one instance of the class is loaded or can be loaded.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 * @static
	 *
	 * @return Elementor_Test_Extension An instance of the class.
	 */
	public static function instance() {

		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self();
		}
		return self::$_instance;

	}

	/**
	 * Constructor
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function __construct() {
		add_action( 'plugins_loaded', [ $this, 'init' ] );

	}


	/**
	 * Initialize the plugin
	 *
	 * Load the plugin only after Elementor (and other plugins) are loaded.
	 * Checks for basic plugin requirements, if one check fail don't continue,
	 * if all check have passed load the files required to run the plugin.
	 *
	 * Fired by `plugins_loaded` action hook.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function init() {

		// Check if Elementor installed and activated
		if ( ! did_action( 'elementor/loaded' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_missing_main_plugin' ] );
			return;
		}

		// Check for required Elementor version
		if ( ! version_compare( ELEMENTOR_VERSION, self::MINIMUM_ELEMENTOR_VERSION, '>=' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_minimum_elementor_version' ] );
			return;
		}

		// Check for required PHP version
		if ( version_compare( PHP_VERSION, self::MINIMUM_PHP_VERSION, '<' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_minimum_php_version' ] );
			return;
		}

		// Add Plugin actions
		add_action( 'elementor/widgets/register', [ $this, 'init_widgets' ] );

		//Register widget style
		add_action( 'elementor/frontend/after_enqueue_styles', [$this, 'widget_styles'] );

        // Register widget scripts
		add_action( 'elementor/frontend/after_enqueue_scripts', [ $this, 'widget_scripts' ]);

        // category register
		add_action( 'elementor/elements/categories_registered',[ $this, 'playrex_elementor_widget_categories' ] );

		// Register Editor scripts
		add_action( 'elementor/editor/after_enqueue_scripts', [$this, 'playrex_editor_scripts'], 100 );

	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have Elementor installed or activated.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function admin_notice_missing_main_plugin() {

		if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

		$message = sprintf(
			/* translators: 1: Plugin name 2: Elementor */
			esc_html__( '"%1$s" requires "%2$s" to be installed and activated.', 'playrex' ),
			'<strong>' . esc_html__( 'Playrex Core', 'playrex' ) . '</strong>',
			'<strong>' . esc_html__( 'Elementor', 'playrex' ) . '</strong>'
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );

	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have a minimum required Elementor version.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function admin_notice_minimum_elementor_version() {

		if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

		$message = sprintf(
			/* translators: 1: Plugin name 2: Elementor 3: Required Elementor version */
			esc_html__( '"%1$s" requires "%2$s" version %3$s or greater.', 'playrex' ),
			'<strong>' . esc_html__( 'Playrex Core', 'playrex' ) . '</strong>',
			'<strong>' . esc_html__( 'Elementor', 'playrex' ) . '</strong>',
			 self::MINIMUM_ELEMENTOR_VERSION
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );

	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have a minimum required PHP version.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function admin_notice_minimum_php_version() {

		if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

		$message = sprintf(
			/* translators: 1: Plugin name 2: PHP 3: Required PHP version */
			esc_html__( '"%1$s" requires "%2$s" version %3$s or greater.', 'playrex' ),
			'<strong>' . esc_html__( 'Playrex Core', 'playrex' ) . '</strong>',
			'<strong>' . esc_html__( 'PHP', 'playrex' ) . '</strong>',
			 self::MINIMUM_PHP_VERSION
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );

	}

	/**
	 * Init Widgets
	 *
	 * Include widgets files and register them
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function init_widgets() {

		// Include Widget files
		require_once( PLAYREX_ADDONS . '/widgets/section-title.php' );
		require_once( PLAYREX_ADDONS . '/widgets/image-box.php' );
		require_once( PLAYREX_ADDONS . '/widgets/playrex-iconbox.php' );
		require_once( PLAYREX_ADDONS . '/widgets/playrex-pricing-box.php' );
		require_once( PLAYREX_ADDONS . '/widgets/playrex-match.php' );
		require_once( PLAYREX_ADDONS . '/widgets/playrex-gallery.php' );
		require_once( PLAYREX_ADDONS . '/widgets/playrex-list-icon.php' );
		require_once( PLAYREX_ADDONS . '/widgets/playrex-button.php' );
		require_once( PLAYREX_ADDONS . '/widgets/playrex-process.php' );
		require_once( PLAYREX_ADDONS . '/widgets/blog-post.php' );
		require_once( PLAYREX_ADDONS . '/widgets/playrex-featured-match.php' );
		require_once( PLAYREX_ADDONS . '/widgets/playrex-player-tab.php' );
		require_once( PLAYREX_ADDONS . '/widgets/playrex-player-card.php' );
		require_once( PLAYREX_ADDONS . '/widgets/playrex-cta.php' );
		require_once( PLAYREX_ADDONS . '/widgets/tournament-schedule-tab.php' );
		require_once( PLAYREX_ADDONS . '/widgets/tournament-schedule.php' );
		require_once( PLAYREX_ADDONS . '/widgets/playrex-team.php' );
		require_once( PLAYREX_ADDONS . '/widgets/playrex-logo.php' );
		require_once( PLAYREX_ADDONS . '/widgets/playrex-faq.php' );
		require_once( PLAYREX_ADDONS . '/widgets/playrex-tournament.php' );
		require_once( PLAYREX_ADDONS . '/widgets/player-3d-slider.php' );
		require_once( PLAYREX_ADDONS . '/widgets/playrex-testimonial.php' );
		require_once( PLAYREX_ADDONS . '/widgets/live-streaming.php' );
		require_once( PLAYREX_ADDONS . '/widgets/upcoming-tournament.php' );
		require_once( PLAYREX_ADDONS . '/widgets/tournament-hero.php' );
		require_once( PLAYREX_ADDONS . '/widgets/tournament-filter.php' );
		require_once( PLAYREX_ADDONS . '/widgets/feature-iconbox.php' );
		require_once( PLAYREX_ADDONS . '/widgets/tournament-history.php' );
		require_once( PLAYREX_ADDONS . '/widgets/match-body.php' );
		require_once( PLAYREX_ADDONS . '/widgets/image-slider.php' );
		require_once( PLAYREX_ADDONS . '/widgets/team-bottominfo.php' );
		require_once( PLAYREX_ADDONS . '/widgets/tournament-body.php' );
		
		
		// header widget
		require_once( PLAYREX_ADDONS . '/header/playrex-new-header.php' );
		require_once( PLAYREX_ADDONS . '/header/playrex-header-top.php' );
		require_once( PLAYREX_ADDONS . '/header/playrex-offcanvas.php' );
		require_once( PLAYREX_ADDONS . '/header/playrex-mobile-menu.php' );
		require_once( PLAYREX_ADDONS . '/header/playrex-megamenu.php' );

		// footer widget
		require_once( PLAYREX_ADDONS . '/footer-widgets/newsletter-widget.php' );

		// Include Woocommerce Widget File
		if( class_exists( 'WooCommerce' ) ){
			require_once( PLAYREX_ADDONS . '/woocommerce/playrex-minicart.php' );
			require_once( PLAYREX_ADDONS . '/woocommerce/playrex-product.php' );
		}
		

		// Register widget
		\Elementor\Plugin::instance()->widgets_manager->register( new \Playrex_Section_Title_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Playrex_Icon_Box() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Playrex_Pricing_Box() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Playrex_Match() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Playrex_Gallery() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Playrex_List_group() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Playrex_Button() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Playrex_Blog_Post() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Playrex_New_Header() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Playrex_Headertop() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Playrex_Offcanvas() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Playrex_Featurmatch() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Playrex_Imagebox() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Playrex_Player_Tab() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Playrex_Player_Card() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Playrex_Cta() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Tournament_Schedule_Tab() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Playrex_Tournament_Schedule() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Playrex_Team() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Playrex_Logo() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Playrex_Mobilemenu() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Playrex_Faq_Area() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Playrex_Tournament() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Playrex_3d_slider() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Playrex_Process() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Playrex_Testimonial() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Playrex_Newsletter_Widgets() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Playrex_Livestreaimg() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Playrex_Upcoming_Tournament() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Playrex_tournament_Hero() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Playrex_tournament_Filter() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Playrex_Feature_Icon_Box() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Playrex_Tournament_History() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Playrex_Match_Body() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Playrex_Image_slider() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Playrex_Bottom_Info() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Playrex_Tournament_Body() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Playrex_Megamenu() );


		// Woocommerce Widget
		if(class_exists( 'WooCommerce' ) ){
			\Elementor\Plugin::instance()->widgets_manager->register( new \Playrex_Minicart() );
			\Elementor\Plugin::instance()->widgets_manager->register( new \Playrex_Product_Slider() );
		}
		
	}

    public function widget_scripts() {
        wp_enqueue_script(
            'playrex-frontend-script',
            PLAYREX_PLUGDIRURI . 'assets/js/playrex-frontend.js',
            array('jquery'),
            time(),
            true
		);
	}

	public function playrex_editor_scripts(){
		wp_enqueue_script(
            'playrex-editor',
            PLAYREX_PLUGDIRURI . 'assets/js/editor.js',
            array('jquery'),
            self::VERSION,
            true
		);
	}

    public function widget_styles() {
        wp_enqueue_style(
            'playrex-widget-style',
            PLAYREX_PLUGDIRURI . 'assets/css/style.css',
			microtime()
		);
	}

    function playrex_elementor_widget_categories( $elements_manager ) {
        $elements_manager->add_category(
            'playrex',
            [
                'title' => __( 'Playrex', 'playrex' ),
                'icon' 	=> 'fa fa-plug',
            ]
        );
        $elements_manager->add_category(
            'playrex_footer_elements',
            [
                'title' => __( 'Playrex Footer Elements', 'playrex' ),
                'icon' 	=> 'fa fa-plug',
            ]
		);

		$elements_manager->add_category(
            'playrex_header_elements',
            [
                'title' => __( 'Playrex Header Elements', 'playrex' ),
                'icon' 	=> 'fa fa-plug',
            ]
        );
	}

}

Playrex_Extension::instance();